import { CopperPositionUserModel } from "../models/copperPositionUserModel";

const copperPositionUserResolvers = {
  Query: {
    /**
     * Get all copper users (active and inactive)
     */
    getAllCopperUsers: async () => {
      try {
        const users = await CopperPositionUserModel.getAllUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch copper users: ${error.message}`);
      }
    },

    /**
     * Get only active copper users
     */
    getActiveCopperUsers: async () => {
      try {
        const users = await CopperPositionUserModel.getAllActiveUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch active copper users: ${error.message}`);
      }
    },

    /**
     * Get user by wallet address
     */
    getCopperUserByWallet: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        const user = await CopperPositionUserModel.findByWallet(wallet_address);
        return user;
      } catch (error: any) {
        throw new Error(`Failed to fetch copper user: ${error.message}`);
      }
    },

    /**
     * Get count of active users
     */
    getActiveCopperUserCount: async () => {
      try {
        const count = await CopperPositionUserModel.countActiveUsers();
        return count;
      } catch (error: any) {
        throw new Error(`Failed to count active copper users: ${error.message}`);
      }
    },
  },

  Mutation: {
    /**
     * Add or update a user
     */
    addCopperUser: async (_: any, { input }: { input: { wallet_address: string } }) => {
      try {
        const { wallet_address } = input;
        
        if (!wallet_address || !wallet_address.match(/^0x[a-fA-F0-9]{40}$/)) {
          return {
            success: false,
            message: "Invalid wallet address format",
            user: null,
          };
        }

        const user = await CopperPositionUserModel.addUser(wallet_address);
        
        return {
          success: true,
          message: "Copper user added/updated successfully",
          user,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to add copper user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Deactivate a user (soft delete)
     */
    deactivateCopperUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await CopperPositionUserModel.deactivateUser(wallet_address);
        
        return {
          success: true,
          message: "Copper user deactivated successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to deactivate copper user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Permanently delete a user
     */
    deleteCopperUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await CopperPositionUserModel.deleteUser(wallet_address);
        
        return {
          success: true,
          message: "Copper user deleted successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to delete copper user: ${error.message}`,
          user: null,
        };
      }
    },
  },
};

export default copperPositionUserResolvers;
