import { GoldPositionUserModel } from "../models/goldPositionUserModel";

const goldPositionUserResolvers = {
  Query: {
    /**
     * Get all gold users (active and inactive)
     */
    getAllGoldUsers: async () => {
      try {
        const users = await GoldPositionUserModel.getAllUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch gold users: ${error.message}`);
      }
    },

    /**
     * Get only active gold users
     */
    getActiveGoldUsers: async () => {
      try {
        const users = await GoldPositionUserModel.getAllActiveUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch active gold users: ${error.message}`);
      }
    },

    /**
     * Get user by wallet address
     */
    getGoldUserByWallet: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        const user = await GoldPositionUserModel.findByWallet(wallet_address);
        return user;
      } catch (error: any) {
        throw new Error(`Failed to fetch gold user: ${error.message}`);
      }
    },

    /**
     * Get count of active users
     */
    getActiveGoldUserCount: async () => {
      try {
        const count = await GoldPositionUserModel.countActiveUsers();
        return count;
      } catch (error: any) {
        throw new Error(`Failed to count active gold users: ${error.message}`);
      }
    },
  },

  Mutation: {
    /**
     * Add or update a user
     */
    addGoldUser: async (_: any, { input }: { input: { wallet_address: string } }) => {
      try {
        const { wallet_address } = input;
        
        if (!wallet_address || !wallet_address.match(/^0x[a-fA-F0-9]{40}$/)) {
          return {
            success: false,
            message: "Invalid wallet address format",
            user: null,
          };
        }

        const user = await GoldPositionUserModel.addUser(wallet_address);
        
        return {
          success: true,
          message: "Gold user added/updated successfully",
          user,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to add gold user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Deactivate a user (soft delete)
     */
    deactivateGoldUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await GoldPositionUserModel.deactivateUser(wallet_address);
        
        return {
          success: true,
          message: "Gold user deactivated successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to deactivate gold user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Permanently delete a user
     */
    deleteGoldUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await GoldPositionUserModel.deleteUser(wallet_address);
        
        return {
          success: true,
          message: "Gold user deleted successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to delete gold user: ${error.message}`,
          user: null,
        };
      }
    },
  },
};

export default goldPositionUserResolvers;
