import { OilPositionUserModel } from "../models/oilPositionUserModel";

const oilPositionUserResolvers = {
  Query: {
    /**
     * Get all oil users (active and inactive)
     */
    getAllOilUsers: async () => {
      try {
        const users = await OilPositionUserModel.getAllUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch oil users: ${error.message}`);
      }
    },

    /**
     * Get only active oil users
     */
    getActiveOilUsers: async () => {
      try {
        const users = await OilPositionUserModel.getAllActiveUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch active oil users: ${error.message}`);
      }
    },

    /**
     * Get user by wallet address
     */
    getOilUserByWallet: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        const user = await OilPositionUserModel.findByWallet(wallet_address);
        return user;
      } catch (error: any) {
        throw new Error(`Failed to fetch oil user: ${error.message}`);
      }
    },

    /**
     * Get count of active users
     */
    getActiveOilUserCount: async () => {
      try {
        const count = await OilPositionUserModel.countActiveUsers();
        return count;
      } catch (error: any) {
        throw new Error(`Failed to count active oil users: ${error.message}`);
      }
    },
  },

  Mutation: {
    /**
     * Add or update a user
     */
    addOilUser: async (_: any, { input }: { input: { wallet_address: string } }) => {
      try {
        const { wallet_address } = input;
        
        if (!wallet_address || !wallet_address.match(/^0x[a-fA-F0-9]{40}$/)) {
          return {
            success: false,
            message: "Invalid wallet address format",
            user: null,
          };
        }

        const user = await OilPositionUserModel.addUser(wallet_address);
        
        return {
          success: true,
          message: "Oil user added/updated successfully",
          user,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to add oil user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Deactivate a user (soft delete)
     */
    deactivateOilUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await OilPositionUserModel.deactivateUser(wallet_address);
        
        return {
          success: true,
          message: "Oil user deactivated successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to deactivate oil user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Permanently delete a user
     */
    deleteOilUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await OilPositionUserModel.deleteUser(wallet_address);
        
        return {
          success: true,
          message: "Oil user deleted successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to delete oil user: ${error.message}`,
          user: null,
        };
      }
    },
  },
};

export default oilPositionUserResolvers;
