import { SilverPositionUserModel } from "../models/silverPositionUserModel";

const silverPositionUserResolvers = {
  Query: {
    /**
     * Get all silver users (active and inactive)
     */
    getAllSilverUsers: async () => {
      try {
        const users = await SilverPositionUserModel.getAllUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch silver users: ${error.message}`);
      }
    },

    /**
     * Get only active silver users
     */
    getActiveSilverUsers: async () => {
      try {
        const users = await SilverPositionUserModel.getAllActiveUsers();
        return users;
      } catch (error: any) {
        throw new Error(`Failed to fetch active silver users: ${error.message}`);
      }
    },

    /**
     * Get user by wallet address
     */
    getSilverUserByWallet: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        const user = await SilverPositionUserModel.findByWallet(wallet_address);
        return user;
      } catch (error: any) {
        throw new Error(`Failed to fetch silver user: ${error.message}`);
      }
    },

    /**
     * Get count of active users
     */
    getActiveSilverUserCount: async () => {
      try {
        const count = await SilverPositionUserModel.countActiveUsers();
        return count;
      } catch (error: any) {
        throw new Error(`Failed to count active silver users: ${error.message}`);
      }
    },
  },

  Mutation: {
    /**
     * Add or update a user
     */
    addSilverUser: async (_: any, { input }: { input: { wallet_address: string } }) => {
      try {
        const { wallet_address } = input;
        
        if (!wallet_address || !wallet_address.match(/^0x[a-fA-F0-9]{40}$/)) {
          return {
            success: false,
            message: "Invalid wallet address format",
            user: null,
          };
        }

        const user = await SilverPositionUserModel.addUser(wallet_address);
        
        return {
          success: true,
          message: "Silver user added/updated successfully",
          user,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to add silver user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Deactivate a user (soft delete)
     */
    deactivateSilverUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await SilverPositionUserModel.deactivateUser(wallet_address);
        
        return {
          success: true,
          message: "Silver user deactivated successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to deactivate silver user: ${error.message}`,
          user: null,
        };
      }
    },

    /**
     * Permanently delete a user
     */
    deleteSilverUser: async (_: any, { wallet_address }: { wallet_address: string }) => {
      try {
        await SilverPositionUserModel.deleteUser(wallet_address);
        
        return {
          success: true,
          message: "Silver user deleted successfully",
          user: null,
        };
      } catch (error: any) {
        return {
          success: false,
          message: `Failed to delete silver user: ${error.message}`,
          user: null,
        };
      }
    },
  },
};

export default silverPositionUserResolvers;
