import React from "react";
import {
  Box,
  Button,
  Dialog,
  DialogContent,
  IconButton,
  Typography,
} from "@mui/material";
import CloseIcon from "@mui/icons-material/Close";

interface ConfirmCloseDialogProps {
  open: boolean;
  title: string;
  description: string;
  confirmLabel?: string;
  cancelLabel?: string;
  onConfirm: () => void;
  onCancel: () => void;
  loading?: boolean;
}

const ConfirmCloseDialog: React.FC<ConfirmCloseDialogProps> = ({
  open,
  title,
  description,
  confirmLabel = "Confirm",
  cancelLabel = "Cancel",
  onConfirm,
  onCancel,
  loading = false,
}) => {
  const handleDialogClose = (
    _event: React.SyntheticEvent,
    reason: "backdropClick" | "escapeKeyDown"
  ) => {
    if (loading) return;
    onCancel();
  };

  return (
    <Dialog
      open={open}
      onClose={handleDialogClose}
      aria-labelledby="close-position-dialog-title"
      PaperProps={{
        sx: {
          borderRadius: "20px",
          p: 4,
          maxWidth: 420,
          width: "100%",
          boxShadow: "0px 24px 48px rgba(15, 23, 42, 0.12)",
        },
      }}
    >
      <Box sx={{ position: "absolute", top: 16, right: 16 }}>
        <IconButton
          aria-label="Close dialog"
          onClick={onCancel}
          size="small"
          disabled={loading}
          sx={{
            color: "#1F2937",
            bgcolor: "transparent",
          }}
        >
          <CloseIcon fontSize="small" />
        </IconButton>
      </Box>
      <DialogContent sx={{ textAlign: "center", px: 2, pt: 2, pb: 0 }}>
        <Typography
          id="close-position-dialog-title"
          variant="h6"
          sx={{
            fontWeight: 700,
            color: "#0F172A",
            mb: 1.5,
          }}
        >
          {title}
        </Typography>
        <Typography
          variant="body1"
          sx={{ color: "#4B5563", mb: 3, lineHeight: 1.6, whiteSpace: "pre-line" }}
        >
          {description}
        </Typography>
        <Box sx={{ display: "flex", flexDirection: "column", gap: 1.5 }}>
          <Button
            onClick={onConfirm}
            variant="contained"
            disableElevation
            disabled={loading}
            sx={{
              py: 1.1,
              fontWeight: 600,
              borderRadius: "999px",
              textTransform: "none",
              fontSize: "1rem",
              color: "#B91C1C",
              backgroundImage:
                "linear-gradient(180deg, rgba(254, 236, 236, 0.96) 0%, rgba(252, 219, 219, 0.96) 100%)",
              border: "1px solid rgba(248, 113, 113, 0.35)",
              boxShadow: "0px 3px 8px rgba(190, 49, 68, 0.16)",
              '&:hover': {
                backgroundImage:
                  "linear-gradient(180deg, rgba(253, 222, 222, 1) 0%, rgba(251, 209, 209, 1) 100%)",
                borderColor: "rgba(220, 38, 38, 0.4)",
                boxShadow: "0px 4px 12px rgba(190, 49, 68, 0.18)",
              },
              '&:active': {
                backgroundImage:
                  "linear-gradient(180deg, rgba(252, 210, 210, 1) 0%, rgba(249, 196, 196, 1) 100%)",
                boxShadow: "0px 2px 6px rgba(190, 49, 68, 0.2)",
              },
              '&:focus-visible': {
                outline: "2px solid rgba(244, 63, 94, 0.45)",
                outlineOffset: "2px",
              },
            }}
          >
            {loading ? "Closing..." : confirmLabel}
          </Button>
          <Button
            onClick={onCancel}
            variant="outlined"
            disabled={loading}
            sx={{
              py: 1.1,
              fontWeight: 600,
              borderRadius: "999px",
              borderColor: "#071C41",
              color: "#071C41",
              textTransform: "none",
              fontSize: "1rem",
              "&:hover": {
                borderColor: "#0B264F",
                backgroundColor: "rgba(7, 28, 65, 0.04)",
              },
            }}
          >
            {cancelLabel}
          </Button>
        </Box>
      </DialogContent>
    </Dialog>
  );
};

export default ConfirmCloseDialog;
