import React, { useState } from "react";
import { useLocation, useNavigate } from "react-router-dom";
import {
  Typography,
  useMediaQuery,
  useTheme,
  Card,
  CardContent,
  Box,
  Button,
  TextField,
} from "@mui/material";
import { ERROR_MESSAGES, TYPOGRAPHY_TEXT,pageData } from "../../const";


const CustomerProductDetails: React.FC = () => {
  const theme = useTheme();
  const isSmallScreen = useMediaQuery(theme.breakpoints.down("md"));
  const location = useLocation();

  const product = location.state?.product || {
    productName: "IntelliWealth Stablecoin",
    productImage: "https://i.imgur.com/uJcFFt2.png",
    assetClass: "Stablecoin",
    minimumInvestment: 1,
    investorType: "Qualified Investor",
    description:`${TYPOGRAPHY_TEXT.MINI_DESCRIPTION}`
     
  };

  const [xdaiAmount, setXdaiAmount] = useState("");
  const [error, setError] = useState("");

  const handleXdaiChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    if (value === "" || parseFloat(value) > 0) {
      setError("");
    } else {
      setError("Please enter a valid amount.");
    }
    setXdaiAmount(value);
  };

  const handleBuyTokens = () => {
    if (xdaiAmount && !error) {
      alert(`Transaction initiated for ${xdaiAmount} IUSD tokens.`);
    }
  };

  if (!product) {
    return (
      <Box
        sx={{
          display: "flex",
          justifyContent: "center",
          alignItems: "center",
          minHeight: "85vh",
          p: 3,
        }}
      >
        <Typography variant="h5" color="error.main">
          {ERROR_MESSAGES.PRODUCT_NOT_FOUND}
        </Typography>
      </Box>
    );
  }

  const InfoRow = ({
    label,
    value,
  }: {
    label: string;
    value: string | number;
  }) => (
    <Box
      sx={{ display: "flex", width: "100%", p: "8px 0", alignItems: "center" }}
    >
      <Typography
        variant="body2"
        sx={{ width: "40%", color: "text.secondary" }}
      >
        {label}
      </Typography>
      <Typography
        variant="body2"
        fontWeight="500"
        sx={{ width: "60%", color: "text.primary" }}
      >
        {value}
      </Typography>
    </Box>
  );

  return (
    <Box sx={{ bgcolor: "#F9FAFC", color: "#333", minHeight: "100vh" }}>
      <Box sx={{ p: isSmallScreen ? 2 : 4 }}>
        <Typography
          variant="h5"
          gutterBottom
          sx={{
            color: "#1A202C",
            fontWeight: 600,
            letterSpacing: "-0.3px",
            padding: "5px",
            textAlign: isSmallScreen ? "center" : "left",
          }}
        >
          {TYPOGRAPHY_TEXT.PRODUCT_DETAILS}
        </Typography>

        <Box
          sx={{
            display: "flex",
            flexDirection: { xs: "column", md: "row" },
            gap: 3,
          }}
        >
          <Card sx={{ flex: 1.5, width: { md: "40%" }, display: "flex", flexDirection: "column" }}>
            <CardContent sx={{ p: { xs: 2, md: 3 } }}>
              <Box
                component="img"
                src={product.productImage}
                alt={`${product.productName} image`}
                sx={{
                  width: "100%",
                  height: "auto",
                  display: "block",
                  borderRadius: "8px",
                  mb: 2,
                }}
              />
              <Typography
                variant="h6"
                component="h2"
                fontWeight="600"
                sx={{ color: "#1A202C", mb: 1 }}
              >
                {product.productName}
              </Typography>
              <Typography
                variant="body2"
                color="text.secondary"
                paragraph
                sx={{ mb: 3 }}
              >
                {product.description}
              </Typography>
              <Box>
                <InfoRow label="Asset Class" value={product.assetClass} />
                <InfoRow
                  label="Min. Investment"
                  value={`$${product.minimumInvestment.toLocaleString()}`}
                />
                <InfoRow label="Investor Type" value={product.investorType} />
              </Box>
            </CardContent>
          </Card>

          <Card sx={{ flex: 1, display: "flex", flexDirection: "column" }}>
            <CardContent
              sx={{
                p: { xs: 2, md: 3 },
                display: "flex",
                flexDirection: "column",
                justifyContent: "center",
                height: "100%",
              }}
            >
              <Box
                sx={{ display: "flex", flexDirection: "column", gap: 2, mb: 4 }}
              >
                <Typography>
                  {TYPOGRAPHY_TEXT.DESCRRIPTION}
                </Typography>
                <Button
                  fullWidth
                  variant="contained"
                  sx={{ textTransform: "none", borderRadius: "8px" }}
                >
                {TYPOGRAPHY_TEXT.TOKENISED_ICO_PAGE}
                </Button>
                <Button
                  fullWidth
                  variant="outlined"
                  sx={{ textTransform: "none", borderRadius: "8px" }}
                >
                 {TYPOGRAPHY_TEXT.IntelliSwap_Exchange}
                </Button>
              </Box>
              <Box>
                <Typography
                  variant="body2"
                  color="text.secondary"
                  sx={{ lineHeight: 1.7 }}
                >
                  {TYPOGRAPHY_TEXT.CONTACT_US}
                  <br /> {TYPOGRAPHY_TEXT.CALL_US_ON}: <strong>{pageData.contact.phone}</strong>
                  <br />  {TYPOGRAPHY_TEXT.EMAIL_US}– <strong>{pageData.contact.email}</strong>
                </Typography>
              </Box>
            </CardContent>
          </Card>
        </Box>

        <Box sx={{ maxWidth: "700px", mx: "auto", my: 6 }}>
          <Card
            sx={{
              bgcolor: "#F5F7FA",
              borderRadius: "12px",
              boxShadow: "none",
              border: "1px solid #E0E0E0",
            }}
          >
            <CardContent sx={{ p: { xs: 2, md: 4 }, textAlign: "center" }}>
              <Typography
                variant="h5"
                component="h3"
                fontWeight="600"
                sx={{ color: "#1A202C", mb: 1 }}
              >
                {pageData.buyInfo.title}
              </Typography>
              <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                {pageData.buyInfo.description}
                <br />
                <strong>{pageData.buyInfo.rate}</strong>
              </Typography>
              <TextField
                fullWidth
                label="Enter xDai Amount"
                variant="outlined"
                value={xdaiAmount}
                onChange={handleXdaiChange}
                type="number"
                error={!!error}
                helperText={error}
                sx={{ mb: 2, maxWidth: "400px", mx: "auto", bgcolor: "#FFF" }}
              />
              <Button
                fullWidth
                variant="contained"
                onClick={handleBuyTokens}
                disabled={!xdaiAmount || !!error}
                sx={{
                  textTransform: "none",
                  fontSize: "16px",
                  py: 1.5,
                  borderRadius: "8px",
                  maxWidth: "400px",
                  mx: "auto",
                }}
              >
                {TYPOGRAPHY_TEXT.BUY_IUSD_TOKENS}
              </Button>
            </CardContent>
          </Card>
        </Box>

        <Box sx={{ mt: 6 }}>
          <Box sx={{ display: "flex", flexWrap: "wrap", gap: 3 }}>
            {pageData.benefits.map((benefit) => (
              <Box
                key={benefit.title}
                sx={{
                  flexGrow: 1,
                  flexBasis: { xs: "100%", sm: "calc(50% - 12px)" },
                  p: 3,
                  bgcolor: "#FFF",
                  border: `1px solid ${theme.palette.divider}`,
                  borderRadius: "12px",
                }}
              >
                <Typography
                  variant="h6"
                  component="h4"
                  fontWeight="600"
                  gutterBottom
                  sx={{ color: "#1A202C" }}
                >
                  {benefit.title}
                </Typography>
                <Typography
                  variant="body2"
                  color="text.secondary"
                  sx={{ lineHeight: 1.6 }}
                >
                  {benefit.description}
                </Typography>
              </Box>
            ))}
          </Box>
        </Box>
      </Box>
    </Box>
  );
};

export default CustomerProductDetails;