import React, { useState } from "react";
import {
  Typography,
  useMediaQuery,
  useTheme,
  Box,
  Button,
  TextField,
  MenuItem,
} from "@mui/material";
import { motion } from "framer-motion";
import goldImage from "../../../assets/Gold.webp";
import iusdImage from "../../../assets/Gold.webp";
import { useNavigate } from "react-router-dom";
import { TEXT } from "../../../const";

const Gold: React.FC = () => {
  const theme = useTheme();
  const isSmallScreen = useMediaQuery(theme.breakpoints.down("md"));
  const navigate = useNavigate();
  const [selectedNetwork, setSelectedNetwork] = useState("Arbitrum Network");
  const networks = ["Arbitrum Network",
    //  "Gnosis Network"
    ] as const;

  const itemVariants = {
    hidden: { opacity: 0, y: 20 },
    visible: {
      opacity: 1,
      y: 0,
      transition: { duration: 0.5 },
    },
  };

  const cardStyles = {
    backgroundColor: "#fff",
    borderRadius: "16px",
    width: isSmallScreen ? "100%" : "360px",
    boxShadow: "0 4px 12px rgba(0, 0, 0, 0.1)",
    transition: "all 0.3s ease",
    minHeight: "730px", 
    display: "flex",
    flexDirection: "column",
    position: "relative",
    overflow: "hidden",
    "&:hover": {
      boxShadow: "0 8px 16px rgba(0, 0, 0, 0.15)",
      transform: "translateY(-6px)",
    },
  };

  const coverImageStyles = {
    width: "100%",
    height: "200px",
    backgroundSize: "cover",
    backgroundPosition: "center",
    borderRadius: "12px 12px 0 0",
    position: "relative",
  };

  const contentStyles = {
    flex: 1,
    display: "flex",
    flexDirection: "column",
    justifyContent: "space-between",
    p: 2,
    pt: 1,
    overflow: "auto",
  };

  const titleStyles = {
    fontWeight: 700,
    fontSize: isSmallScreen ? "22px" : "26px",
color: "#0b3d91",
    mb: 1.5,
  };

  const descriptionStyles = {
    fontWeight: 400,
    fontSize: isSmallScreen ? "14px" : "16px",
    color: "#455a64",
    mb: 2,
    lineHeight: 1.5,
    textAlign: "left",
  };

  const detailLabelStyles = {
    fontWeight: 500,
    fontSize: isSmallScreen ? "14px" : "15px",
    color: "#455a64",
  };

  const detailValueStyles = {
    fontWeight: 700,
    fontSize: isSmallScreen ? "14px" : "15px",
    color: "#455a64",
  };

  const buttonStyles = {
    color: "#fff",
    px: 3,
    py: 1.5,
    textTransform: "none",
    fontSize: isSmallScreen ? "14px" : "16px",
    borderRadius: "8px",
    width: "100%",
    bgcolor: "#0b3d91",
    '&:hover': { bgcolor: "#083475", },
    mt: "auto",
  };

  const textFieldStyles = {
    width: "100%",
    "& .MuiInputBase-input": {
      color: "#455a64",
    },
    "& .MuiOutlinedInput-root": {
      "& fieldset": {
        borderColor: "#90a4ae",
      },
      "&:hover fieldset": {
        borderColor: "#1976d2",
      },
      "&.Mui-focused fieldset": {
        borderColor: "#1976d2",
      },
    },
  };

  const cards = [
    {
      title: "IntelliGold Futures Order-Book & Marketplace",
      description: "A decentralised platform providing a comprehensive order book and marketplace for creating, subscribing to, and trading tokenised gold futures. It allows for physical settlement upon contract maturity.",
      assetClass: "Tokenised Futures",
      minInvestment: "$10,000",
      investorType: "Accredited Investor",
      settlement: "Digital / Physical",
      buttonText: "View Exchange",
      navigateTo: "/markets/commodities/gold/orderbookgold",
      showNetworkField: false,
      image: goldImage,
    },
    {
      title: TEXT.INTELLITRADE_GOLD_FUTURES_OTC_RFQ,
      description: TEXT.INTELLITRADE_GOLD_FUTURES_OTC_RFQ_DESCRIPTION,
      assetClass: TEXT.OTC_DERIVATIVES,
      minInvestment: "$50,000",
      investorType: TEXT.INSTITUTIONAL,
      keyFeatures: TEXT.CALENDAR_SPREADS,
      buttonText: TEXT.LAUNCH_OTC_DESK,
      navigateTo: "/markets/commodities/gold/intellitradegold",
      showNetworkField: true,
      image: goldImage,
    },
    {
      title: TEXT.INTELLIWEALTH_GOLD_IUSD,
      description: "Acquire asset-backed gold tokens (RGOLD) from professionally managed primary sales. Purchase digital assets directly backed by verified, audited gold reserves at a discounted price",
      assetClass: "Spot Tokenised Asset",
      minInvestment: "No Minimum",
      investorType: "Retail / Professional",
      settlement: "Digital (RGOLD Token)",
      buttonText: "View Token Sale",
      navigateTo: "/markets/commodities/gold/iusdacquire",
      showNetworkField: false,
      image: iusdImage,
    },
  ];

  return (
    <Box
      sx={{
        display: "flex",
        flexDirection: "column",
        alignItems: "flex-start",
        bgcolor: "#F9FAFC",
        minHeight: "85vh",
        p: { xs: 2, md: 4 },
      }}
    >
      <Typography
        variant="h4"
        sx={{
          color: "#1A202C",
          fontWeight: 700,
          letterSpacing: "-0.5px",
          mb: 4,
          textAlign: "left",
        }}
      >
        Commodities Gold
      </Typography>
      <Box
        sx={{
          display: "flex",
          flexDirection: isSmallScreen ? "column" : "row",
          gap: 2,
          overflowX: isSmallScreen ? "visible" : "auto",
        }}
      >
        {cards.map((card, index) => (
          <motion.div
            key={index}
            variants={itemVariants}
            initial="hidden"
            animate="visible"
            style={{ width: isSmallScreen ? "100%" : "360px", flexShrink: 0 }}
          >
            <Box sx={cardStyles}>
              <Box sx={{ ...coverImageStyles, backgroundImage: `url(${card.image})`, backgroundColor: "#1a1a2e" }} />
              <Box sx={contentStyles}>
                <Box>
                  <Typography sx={titleStyles}>{card.title}</Typography>
                  <Typography sx={descriptionStyles}>
                    {card.description}
                  </Typography>
                  <Box
                    sx={{
                      display: "flex",
                      justifyContent: "space-between",
                      mb: 1,
                    }}
                  >
                    <Box>
                      <Typography sx={detailLabelStyles}>
                        {TEXT.ASSET_CLASS}
                      </Typography>
                      <Typography sx={detailValueStyles}>
                        {card.assetClass}
                      </Typography>
                    </Box>
                    <Box sx={{ textAlign: "right" }}>
                      <Typography sx={detailLabelStyles}>
                        {TEXT.MIN_INVESTMENT || "Min. Investment"}
                      </Typography>
                      <Typography sx={detailValueStyles}>
                        {card.minInvestment}
                      </Typography>
                    </Box>
                  </Box>
                  <Box
                    sx={{
                      display: "flex",
                      justifyContent: "space-between",
                      mb: 1,
                    }}
                  >
                    <Box>
                      <Typography sx={detailLabelStyles}>
                        {TEXT.INVESTOR_TYPE}
                      </Typography>
                      <Typography sx={detailValueStyles}>
                        {card.investorType}
                      </Typography>
                    </Box>
                    <Box sx={{ textAlign: "right" }}>
                      <Typography sx={detailLabelStyles}>
                        {card.keyFeatures ? TEXT.KEY_FEATURES : TEXT.SETTLEMENT}
                      </Typography>
                      <Typography sx={detailValueStyles}>
                        {card.keyFeatures || card.settlement}
                      </Typography>
                    </Box>
                  </Box>
                  {card.showNetworkField ? (
                    <Box mt={3}>
                      <Typography sx={{ ...detailLabelStyles, mb: 1 }}>
                        {TEXT.SELECT_NETWORK}
                      </Typography>
                      <TextField
                        select
                        value={selectedNetwork}
                        onChange={(e) => setSelectedNetwork(e.target.value)}
                        variant="outlined"
                        size="small"
                        sx={textFieldStyles}
                        fullWidth
                      >
                        {networks.map((net) => (
                          <MenuItem key={net} value={net}>
                            {net}
                          </MenuItem>
                        ))}
                      </TextField>
                    </Box>
                  ) : (
                    <Box mt={3} sx={{ height: "64px" }} /> 
                  )}
                </Box>
                <Button
                  variant="contained"
                  sx={buttonStyles}
                  aria-label={card.buttonText}
                  onClick={() => {
                    if (card.navigateTo.includes("/intellitradegold")) {
                      if (selectedNetwork === "Gnosis Network") {
                        navigate("/markets/commodities/gold/intellitradegold-gnosis");
                      } else {
                        navigate(card.navigateTo);
                      }
                    } else {
                      navigate(card.navigateTo);
                    }
                  }}
                >
                  {card.buttonText}
                </Button>
              </Box>
            </Box>
          </motion.div>
        ))}
      </Box>
    </Box>
  );
};

export default Gold;
