import React, {
  useState,
  useEffect,
  useRef,
  useCallback,
  ChangeEvent,
  useMemo,
} from "react";
import {
  ethers,
  BrowserProvider,
  JsonRpcSigner,
  Contract,
  MaxUint256,
  Eip1193Provider,
} from "ethers";
import Chart from "chart.js/auto";
import type { TransactionRequest, TransactionReceipt } from "ethers";
import { SelectChangeEvent } from "@mui/material/Select";
import {
  Box,
  Typography,
  Paper,
  Button,
  CircularProgress,
  Snackbar,
  Alert,
  TextField,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Chip,
  Tabs,
  Tab,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Tooltip,
  FormHelperText,
  Divider,
  Checkbox,
  FormControlLabel,
} from "@mui/material";
import Slider from "@mui/material/Slider";
import { useTheme } from "@mui/material/styles";
import { Logout as LogoutIcon } from "@mui/icons-material";
import TaskAltIcon from "@mui/icons-material/TaskAlt";
import { useQuery } from "@apollo/client";
import { GET_CUSTOMER_ID } from "../../../graphql/query";
import ConfirmCloseDialog from "../../../components/ConfirmCloseDialog";
import { PRODUCTS } from "../../../const";

const ADDRESS_ZERO = "0x0000000000000000000000000000000000000000";

// Constants from the script
const RFQ_ADDR = "0xaCD4995aF6919Abe8BeAb275A13Bd925238c3bA5";
const IGLD_ADDR = "0xC15Dc4e6607E031ab5D81B70A74B6A62402f5237";
const IUSD_ADDR = "0xFEF7D9E974713E56Be167C8685e253DE5ec9c8d0";
const RFQ_ABI = PRODUCTS.GOLD.RFQ_ABI;
const ERC20_ALLOWANCE_ABI = [
  {
    inputs: [
      { internalType: "address", name: "owner", type: "address" },
      { internalType: "address", name: "spender", type: "address" },
    ],
    name: "allowance",
    outputs: [{ internalType: "uint256", name: "", type: "uint256" }],
    stateMutability: "view",
    type: "function",
  },
  {
    inputs: [
      { internalType: "address", name: "spender", type: "address" },
      { internalType: "uint256", name: "value", type: "uint256" },
    ],
    name: "approve",
    outputs: [{ internalType: "bool", name: "", type: "bool" }],
    stateMutability: "nonpayable",
    type: "function",
  },
];

// Interfaces for types
interface Leg {
  maturity: string; // formatted as DD/MM/YYYY
  amountIGLD: number;
  isLong: boolean;
  entryPrice: number;
}

interface Position {
  positionId: string;
  positionType: string;
  legs: Leg[];
  longAmount: number;
  stopLossPrice: number;
  takeProfitPrice: number;
  pnl: number;
  currentPrice?: number | null;
}

interface PortfolioState {
  positions: Position[];
  totalValue: number;
  pnl: number;
  activePositions: number;
  spotPrice: number;
  isLoading: boolean;
}

interface OutrightTrade {
  maturity: string;
  amount: string;
  sl: string;
  tp: string;
  useSlTp: boolean;
  paymentToken: string;
  maxCost: string;
  slippagePercent: string;
}

interface SpreadLeg {
  maturity: string;
  amountIGLD: string;
  isLong: boolean;
}

interface SpreadTrade {
  legs: SpreadLeg[];
  paymentToken: string;
  maxCost: string;
  minCredit: string;
  slippagePercent: string;
  impliedVolatility: string;
  ivEnabled: boolean;
}

interface TradeState {
  outright: OutrightTrade;
  spread: SpreadTrade;
}

interface SpreadQuoteState {
  isCalculating: boolean;
  netCost: bigint | null;
  displayCost: string;
  error: string | null;
}

interface AdminState {
  manualSpotPrice: string;
  keeperAddress: string;
  annualRate: string;
  oracleMaxAge: string;
}

export const freezeNowSec = () => Math.floor(Date.now() / 1000);

export const yearsToMaturity = (maturityTs: number, nowSec: number) =>
  (maturityTs - nowSec) / (365 * 24 * 60 * 60);

interface AnalysisState {
  startDate: string;
  endDate: string;
  results:
    | {
        priceSpread: number;
        basisPoints: number;
        annualizedYield: number;
        startEndPrice: string;
      }
    | null
    | "loading";
}

interface StatusState {
  type: "disconnected" | "error" | "connected";
  message: string;
}

interface ToastState {
  visible: boolean;
  message: string;
  type: "info" | "success" | "error" | "warning";
}

// Utility functions
const formatCurrency = (value: number, decimals: number = 2): string => {
  if (isNaN(value) || !isFinite(value)) return "$0.00";
  return `$${value.toFixed(decimals)}`;
};

const formatPnlWithSign = (pnl: number): string => {
  if (isNaN(pnl)) return "+$0.00";
  const sign = pnl >= 0 ? "+" : "-";
  const formatted = Math.abs(pnl).toLocaleString("en-GB", {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  });
  return `${sign}$${formatted}`;
};

const getPnlColor = (value: number) =>
  value >= 0 ? "#10B981" : "#EF4444";

const formatSLTP = (price: number, type: string): string => {
  if (price > 0) return `${type}: ${formatCurrency(price)}`;
  return `${type}: --`;
};

const renderPositionTypeChip = (type: string) => {
  const label = type || "Unknown";
  const isOutright = label.toLowerCase() === "outright";
  return (
    <Chip
      label={label}
      size="small"
      aria-label={`Position type: ${label}`}
      sx={{
        borderRadius: "999px",
        fontWeight: 600,
        bgcolor: isOutright
          ? "rgba(96, 165, 250, 0.2)"
          : "rgba(196, 181, 253, 0.25)",
        color: isOutright ? "#1D4ED8" : "#7C3AED",
      }}
    />
  );
};

// Format a Date into DD/MM/YYYY
const formatDateDDMMYYYY = (d: Date): string => {
  const dd = String(d.getDate()).padStart(2, "0");
  const mm = String(d.getMonth() + 1).padStart(2, "0");
  const yyyy = d.getFullYear();
  return `${dd}/${mm}/${yyyy}`;
};

const shortenAddress = (address: string): string => {
  if (!address) return "";
  return `${address.slice(0, 8)}...${address.slice(-5)}`;
};

// Extract a concise error message (aligns with gold.html behavior)
const extractErrorMessage = (err: any): string => {
  try {
    if (!err) return "An unknown error occurred.";
    const msg =
      err?.reason ||
      err?.data?.message ||
      err?.data?.originalError?.message ||
      err?.error?.message ||
      err?.error?.data?.message ||
      err?.error?.data?.originalError?.message ||
      err?.info?.error?.message ||
      err?.info?.error?.data?.message ||
      err?.info?.error?.data?.originalError?.message ||
      err?.shortMessage ||
      err?.message ||
      String(err);
    return typeof msg === "string" ? msg : JSON.stringify(msg);
  } catch {
    return "An unknown error occurred.";
  }
};

// Search deeply for an "execution reverted" marker within common error shapes
const containsExecutionReverted = (err: any, fallbackMsg?: string): boolean => {
  const haystacks: string[] = [];
  try {
    if (fallbackMsg) haystacks.push(fallbackMsg);
    if (typeof err === "string") haystacks.push(err);
    if (err?.reason) haystacks.push(err.reason);
    if (err?.message) haystacks.push(err.message);
    if (err?.shortMessage) haystacks.push(err.shortMessage);
    if (err?.data?.message) haystacks.push(err.data.message);
    if (err?.data?.originalError?.message)
      haystacks.push(err.data.originalError.message);
    if (err?.error?.message) haystacks.push(err.error.message);
    if (err?.error?.data?.message) haystacks.push(err.error.data.message);
    if (err?.error?.data?.originalError?.message)
      haystacks.push(err.error.data.originalError.message);
    if (err?.info?.error?.message) haystacks.push(err.info.error.message);
    if (err?.info?.error?.data?.message)
      haystacks.push(err.info.error.data.message);
    if (err?.info?.error?.data?.originalError?.message)
      haystacks.push(err.info.error.data.originalError.message);

    // Ethers/MetaMask sometimes include a JSON string in .body
    const bodies = [err?.body, err?.error?.body, err?.info?.body].filter(
      Boolean
    );
    for (const body of bodies) {
      if (typeof body === "string") {
        try {
          const parsed = JSON.parse(body);
          const nested = [
            parsed?.error?.message,
            parsed?.error?.data?.message,
            parsed?.error?.data?.originalError?.message,
          ].filter(Boolean);
          haystacks.push(...(nested as string[]));
        } catch (_) {
          haystacks.push(body);
        }
      }
    }
  } catch {
    // ignore
  }
  return haystacks.some(
    (s) =>
      typeof s === "string" && s.toLowerCase().includes("execution reverted")
  );
};

const CLOSE_POSITION_INSUFFICIENT_FUNDS_SELECTOR = "0xe450d38c";

const extractRevertHex = (err: any): string | null => {
  const candidates = [
    err?.data?.data,
    err?.data,
    err?.error?.data?.data,
    err?.error?.data,
    err?.info?.error?.data?.data,
    err?.info?.error?.data,
  ];

  for (const value of candidates) {
    if (typeof value === "string" && value.startsWith("0x") && value.length > 10) {
      return value;
    }
  }

  const bodies = [err?.body, err?.error?.body, err?.info?.body].filter(Boolean);
  for (const body of bodies) {
    if (typeof body !== "string") continue;
    try {
      const parsed = JSON.parse(body);
      const nested = [
        parsed?.error?.data?.data,
        parsed?.error?.data,
      ];
      for (const value of nested) {
        if (typeof value === "string" && value.startsWith("0x") && value.length > 10) {
          return value;
        }
      }
    } catch {
      // ignore malformed JSON bodies
    }
  }

  return null;
};

const interpretClosePositionLiquidityError = (err: any): string | null => {
  const hexData = extractRevertHex(err);
  if (!hexData || !hexData.startsWith(CLOSE_POSITION_INSUFFICIENT_FUNDS_SELECTOR)) {
    return null;
  }

  try {
    const abi = ethers.AbiCoder.defaultAbiCoder();
    const decoded = abi.decode(
      ["address", "uint256", "uint256"],
      `0x${hexData.slice(10)}`
    );
    const balanceWei = decoded[1] as bigint;
    const expectedWei = decoded[2] as bigint;

    const expected = ethers.formatUnits(expectedWei, 18);
    const shortfallWei =
      expectedWei > balanceWei ? expectedWei - balanceWei : BigInt(0);
    const shortfall = ethers.formatUnits(shortfallWei, 18);

    if (shortfallWei > BigInt(0)) {
      return `Close failed: the RFQ treasury needs ${shortfall} more IUSD (payout requires ${expected} IUSD). Please contact support.`;
    }
    return `Close failed: the RFQ treasury lacks sufficient IUSD liquidity (required payout ${expected} IUSD).`;
  } catch {
    return "Close failed: the RFQ treasury lacks sufficient IUSD liquidity.";
  }
};

// Black-Scholes implementation for IV simulation
const cdf = (x: number): number => {
  const a1 = 0.254829592;
  const a2 = -0.284496736;
  const a3 = 1.421413741;
  const a4 = -1.453152027;
  const a5 = 1.061405429;
  const p = 0.3275911;

  let sign = 1;
  if (x < 0) sign = -1;
  const absX = Math.abs(x) / Math.sqrt(2.0);

  const t = 1.0 / (1.0 + p * absX);
  const y =
    1.0 -
    ((((a5 * t + a4) * t + a3) * t + a2) * t + a1) * t * Math.exp(-absX * absX);

  return 0.5 * (1.0 + sign * y);
};

const blackScholes = (
  callOrPut: "call" | "put",
  S: number,
  K: number,
  t: number,
  r: number,
  sigma: number
): number => {
  if (sigma <= 0 || t <= 0) {
    return 0;
  }

  const d1 =
    (Math.log(S / K) + (r + (sigma * sigma) / 2) * t) / (sigma * Math.sqrt(t));
  const d2 = d1 - sigma * Math.sqrt(t);

  if (callOrPut === "call") {
    return S * cdf(d1) - K * Math.exp(-r * t) * cdf(d2);
  } else if (callOrPut === "put") {
    return K * Math.exp(-r * t) * cdf(-d2) - S * cdf(-d1);
  }
  return 0;
};
export const applyIvAndToWei = (
  basePriceWei: bigint,
  tYears: number,
  annualRate: number,
  volatility: number,
  blackScholes: (
    type: "call" | "put",
    S: number,
    K: number,
    t: number,
    r: number,
    v: number
  ) => number
): bigint => {
  const base = parseFloat(ethers.formatUnits(basePriceWei, 18));
  const prem = blackScholes("call", base, base, tYears, annualRate, volatility);
  const simulated = base + prem;
  const rounded = simulated.toFixed(18);
  return ethers.parseUnits(rounded, 18);
};

export const BIGINT_0 = BigInt(0);
export const BIGINT_10000 = BigInt(10000);

export const formatCurrencyIN = (value: number, decimals: number = 2) => {
  if (!isFinite(value)) return "$0.00";
  return `$${value.toLocaleString("en-GB", {
    minimumFractionDigits: decimals,
    maximumFractionDigits: decimals,
  })}`;
};

export const applySlippageMaxCost = (netCostWei: bigint, bps: bigint) =>
  (netCostWei * (BIGINT_10000 + bps)) / BIGINT_10000;

export const applySlippageMinCredit = (creditAbsWei: bigint, bps: bigint) =>
  (creditAbsWei * (BIGINT_10000 - bps)) / BIGINT_10000;

const IntelliTradeGoldWithoutSLTP = () => {
  const theme = useTheme();
  const [status, setStatus] = useState<StatusState>({
    type: "disconnected",
    message: "Not Connected",
  });
  const [activeTab, setActiveTab] = useState<"portfolio" | "trade" | "admin">(
    "portfolio"
  );
  const [userAddress, setUserAddress] = useState<string>("");
  const [isAdmin, setIsAdmin] = useState<boolean>(false);
  const [provider, setProvider] = useState<BrowserProvider | null>(null);
  const [signer, setSigner] = useState<JsonRpcSigner | null>(null);
  const [contract, setContract] = useState<Contract | null>(null);
  const storedUser = JSON.parse(localStorage.getItem("user") || "{}");
  const id = storedUser?.id || "";
  const { loading, error, data, refetch } = useQuery(GET_CUSTOMER_ID, {
    variables: { id },
  });
  const customerData = data?.getCustomer || {};
  const [toast, setToast] = useState<ToastState>({
    visible: false,
    message: "",
    type: "info",
  });
  const [oracleIsStale, setOracleIsStale] = useState<boolean>(false);
  const [isManualSource, setIsManualSource] = useState<boolean>(false);
  const [portfolio, setPortfolio] = useState<PortfolioState>({
    positions: [],
    totalValue: 0,
    pnl: 0,
    activePositions: 0,
    spotPrice: 0,
    isLoading: false,
  });
  const [trade, setTrade] = useState<TradeState>({
    outright: {
      maturity: "",
      amount: "",
      sl: "",
      tp: "",
      useSlTp: false,
      paymentToken: IUSD_ADDR,
      maxCost: "0.00",
      slippagePercent: "0.5",
    },
    spread: {
      legs: [
        { maturity: "", amountIGLD: "", isLong: true },
        { maturity: "", amountIGLD: "", isLong: false },
      ],
      paymentToken: IUSD_ADDR,
      maxCost: "",
      minCredit: "",
      slippagePercent: "0.5",
      impliedVolatility: "25.0",
      ivEnabled: true,
    },
  });
  const [spreadQuote, setSpreadQuote] = useState<SpreadQuoteState>({
    isCalculating: false,
    netCost: null,
    displayCost: "$0.00",
    error: null,
  });
  const [admin, setAdmin] = useState<AdminState>({
    manualSpotPrice: "",
    keeperAddress: "",
    annualRate: "",
    oracleMaxAge: "",
  });
  const [analysis, setAnalysis] = useState<AnalysisState>({
    startDate: "",
    endDate: "",
    results: null,
  });
  const [isOutrightTrading, setIsOutrightTrading] = useState<boolean>(false);
  const [isSpreadTrading, setIsSpreadTrading] = useState<boolean>(false);
  const [pendingClosePosition, setPendingClosePosition] =
    useState<Position | null>(null);
  const [isCloseProcessing, setIsCloseProcessing] = useState(false);
  const autoCloseInFlightRef = useRef<Set<string>>(new Set());
  const autoCloseLastAttemptRef = useRef<Record<string, number>>({});
  const [closeRefreshToken, setCloseRefreshToken] = useState(0);

  const chartRef = useRef<HTMLCanvasElement | null>(null);
  const chartInstance = useRef<Chart<"line", number[], string> | null>(null);
  const lastOutrightSigRef = useRef<string>("");
  const lastSpreadSigRef = useRef<string>("");
  const quoteNowRef = useRef<number>(0);

  const isUserRejectedError = (err: any) => {
    const code = err?.code ?? err?.error?.code ?? err?.error?.error?.code;
    return code === 4001 || code === "ACTION_REJECTED";
  };

  const getUserRejectedMessage = (err: any) => {
    const detail =
      err?.error?.data?.message ||
      err?.data?.message ||
      err?.error?.message ||
      err?.message;

    if (typeof detail === "string") {
      const lower = detail.toLowerCase();
      if (lower.includes("user denied") || lower.includes("user rejected")) {
        return "Transaction cancelled in MetaMask (user denied signature).";
      }
      if (lower.includes("rejected")) {
        return "Transaction cancelled in MetaMask.";
      }
    }

    return "Transaction cancelled in MetaMask.";
  };

  // Error handling
  const handleError = useCallback(
    (e: Error, context: string = "General") => {
      console.error(`${context} error:`, e);
      const message = extractErrorMessage(e);

      if (message.includes("Oracle price is stale")) {
        if (!oracleIsStale) {
          setToast({
            visible: true,
            message: "Oracle is stale. Functionality is limited.",
            type: "error",
          });
          setTimeout(
            () => setToast((prev) => ({ ...prev, visible: false })),
            10000
          );
        }
        setOracleIsStale(true);
        return;
      }

      // Special-case: show only the concise maturity error
      try {
        const compact = message
          .replace(/^["']|["']$/g, "")
          .replace(/.*execution reverted:\s*/i, "")
          .trim();
        const hay = `${message} ${compact} ${JSON.stringify((e as any) ?? {})}`;
        if (/maturity\s+too\s+soon/i.test(hay)) {
          setToast({
            visible: true,
            message: "Maturity too soon",
            type: "error",
          });
          setTimeout(
            () => setToast((prev) => ({ ...prev, visible: false })),
            8000
          );
          return;
        }
      } catch {}

      // Prefer concise messages for common cases
      const lower = (message || "").toLowerCase();
      const conciseMsg = lower.includes("no matching fragment")
        ? "Error: Net short positions are not supported. Total long amount must be >= total short amount."
        : containsExecutionReverted(e, message)
        ? "Transaction error: execution reverted"
        : `${context} failed: ${message.substring(0, 100)}`;

      setToast({ visible: true, message: conciseMsg, type: "error" });
      setTimeout(() => setToast((prev) => ({ ...prev, visible: false })), 8000);
    },
    [oracleIsStale]
  );

  // Toast notification
  const showToast = useCallback(
    (
      message: string,
      type: ToastState["type"] = "info",
      duration: number = 4000
    ) => {
      setToast({ visible: true, message, type });
      if (duration) {
        setTimeout(
          () => setToast((prev) => ({ ...prev, visible: false })),
          duration
        );
      }
    },
    []
  );

  const [isLoading, setIsLoading] = useState(false);

  const [balance, setBalance] = useState<string | null>(null);
  interface CustomEthereumProvider extends Eip1193Provider {
    on(
      event: "accountsChanged" | "chainChanged",
      listener: (...args: any[]) => void
    ): void;
    removeListener(
      event: "accountsChanged" | "chainChanged",
      listener: (...args: any[]) => void
    ): void;
  }

  const checkIfWalletIsConnected = async (
    provider: BrowserProvider
  ): Promise<string | null> => {
    try {
      const accounts = await provider.send("eth_accounts", []);
      return accounts.length > 0 ? accounts[0] : null;
    } catch (e) {
      console.error("Error checking wallet connection:", e);
      return null;
    }
  };

  const connectWallet = useCallback(async () => {
    setIsLoading(true);
    try {
      const eth = (window as any).ethereum as Eip1193Provider | undefined;
      if (!eth) {
        showToast("Please install a web3 wallet like MetaMask.", "error", 5000);
        setStatus({ type: "error", message: "No wallet detected" });
        return;
      }

      const provider = new BrowserProvider(eth as Eip1193Provider, "any");

      const connectedAccount = await checkIfWalletIsConnected(provider);
      if (connectedAccount) {
        const signer = await provider.getSigner();
        const network = await provider.getNetwork();

        if (Number(network.chainId) !== 42161) {
          showToast("Please switch to Arbitrum One.", "error", 5000);
          setStatus({
            type: "error",
            message: "Please switch to Arbitrum One",
          });
          setIsLoading(false);
          return;
        }

        const contract = new Contract(RFQ_ADDR, RFQ_ABI, signer);
        const balance = await provider.getBalance(connectedAccount);
        const ethBalance = ethers.formatEther(balance);

        setProvider(provider);
        setSigner(signer);
        setUserAddress(connectedAccount);
        setContract(contract);
        setBalance(ethBalance);
        setStatus({
          type: "connected",
          message: `Connected: ${shortenAddress(connectedAccount)}`,
        });

        const oracleOK = await checkOracleStatus();
        await reloadData();
        setIsLoading(false);
        return;
      }

      await new Promise((resolve) => setTimeout(resolve, 500));
      const accounts = await provider.send("eth_requestAccounts", []);
      if (accounts.length === 0) {
        showToast("No accounts found. Please unlock MetaMask.", "error", 5000);
        setStatus({ type: "error", message: "No accounts found" });
        setIsLoading(false);
        return;
      }

      const network = await provider.getNetwork();
      if (Number(network.chainId) !== 42161) {
        showToast("Please switch to Arbitrum One.", "error", 5000);
        setStatus({ type: "error", message: "Please switch to Arbitrum One" });
        setIsLoading(false);
        return;
      }

      const signer = await provider.getSigner();
      const userAddress = accounts[0];
      const contract = new Contract(RFQ_ADDR, RFQ_ABI, signer);
      const balance = await provider.getBalance(userAddress);
      const ethBalance = ethers.formatEther(balance);

      setProvider(provider);
      setSigner(signer);
      setUserAddress(userAddress);
      setContract(contract);
      setBalance(ethBalance);
      setStatus({
        type: "connected",
        message: `Connected: ${shortenAddress(userAddress)}`,
      });

      const oracleOK = await checkOracleStatus();
      await reloadData();
      setIsLoading(false);
    } catch (e: unknown) {
      if ((e as any).code === -32002) {
        showToast(
          "MetaMask is processing another request. Please complete or cancel it in MetaMask and try again.",
          "error",
          8000
        );
        setStatus({ type: "error", message: "MetaMask request pending" });
      } else {
        handleError(e as Error, "Connection");
      }
      setIsLoading(false);
    }
  }, [handleError, showToast]);

  // Disconnect wallet function
  const disconnectWallet = useCallback(() => {
    setStatus({ type: "disconnected", message: "Not Connected" });
    setUserAddress("");
    setProvider(null);
    setSigner(null);
    setContract(null);
    setBalance(null);
    setPortfolio({
      positions: [],
      totalValue: 0,
      pnl: 0,
      activePositions: 0,
      spotPrice: 0,
      isLoading: false,
    });
    setIsAdmin(false);
    showToast("Wallet disconnected.", "info", 4000);
  }, [showToast]);

  // Effect for wallet connection and event listeners
  useEffect(() => {
    const ethAny = (window as any).ethereum as
      | CustomEthereumProvider
      | undefined;
    if (!ethAny) {
      showToast("Please install a web3 wallet like MetaMask.", "error", 5000);
      return;
    }

    const customEthereum = ethAny;

    const handleAccountsChanged = (accounts: string[]) => {
      if (accounts.length > 0) {
        setUserAddress(accounts[0]);
        const provider = new BrowserProvider(customEthereum);
        provider.getBalance(accounts[0]).then((balance: bigint) => {
          setBalance(ethers.formatEther(balance));
        });
        setStatus({
          type: "connected",
          message: `Connected: ${shortenAddress(accounts[0])}`,
        });
        reloadData();
      } else {
        disconnectWallet();
      }
    };

    const handleChainChanged = () => {
      window.location.reload();
    };

    customEthereum.on("accountsChanged", handleAccountsChanged);
    customEthereum.on("chainChanged", handleChainChanged);

    // Initial connection attempt
    connectWallet();

    return () => {
      customEthereum.removeListener("accountsChanged", handleAccountsChanged);
      customEthereum.removeListener("chainChanged", handleChainChanged);
    };
  }, [connectWallet, disconnectWallet, showToast]);

  // Effect to fetch data when userAddress changes
  useEffect(() => {
    if (userAddress && contract) {
      checkAdminStatus();
      reloadData();
    }
  }, [userAddress, contract]);
  // Effect to fetch data when userAddress changes
  useEffect(() => {
    if (userAddress && contract) {
      checkAdminStatus();
      reloadData();
    }
  }, [userAddress, contract]);

  // Check oracle status
  const checkOracleStatus = useCallback(async (): Promise<boolean> => {
    if (!contract) return false;
    try {
      const source = (await contract.spotSource()) as bigint;
      setIsManualSource(source === BigInt(1));

      if (source === BigInt(1)) {
        setOracleIsStale(false);
        console.log("Oracle status: Manual source active");
        return true;
      }

      await contract.priceFor(Math.floor(Date.now() / 1000) + 86400);
      setOracleIsStale(false);
      console.log("Oracle status: Chainlink active and up-to-date");
      return true;
    } catch (e: unknown) {
      console.error("Oracle check failed:", e);
      handleError(e as Error, "Oracle Status Check");
      return false;
    }
  }, [contract, handleError]);

  // Fetch live spot price
  const fetchLiveSpotPrice = useCallback(async () => {
    if (!contract || (oracleIsStale && !isManualSource)) {
      console.log("Skipping spot price fetch due to stale oracle");
      return;
    }
    try {
      const price = (await contract.priceFor(
        Math.floor(Date.now() / 1000)
      )) as bigint;
      setPortfolio((prev) => ({
        ...prev,
        spotPrice: parseFloat(ethers.formatUnits(price, 18)),
      }));
    } catch (e: unknown) {
      console.error("Spot price fetch error:", e);
      handleError(e as Error, "Live Spot Price");
      throw e;
    }
  }, [contract, oracleIsStale, isManualSource, handleError]);

  // Fetch admin settings
  const fetchAdminSettings = useCallback(async () => {
    if (!contract || !isAdmin) return;
    try {
      const [maxAge, annualRate, priceSrc] = await Promise.all([
        contract.oracleMaxAge() as Promise<bigint>,
        contract.annualRateBPS() as Promise<bigint>,
        contract.spotSource() as Promise<bigint>,
      ]);
      setAdmin((prev) => ({
        ...prev,
        oracleMaxAge: maxAge.toString(),
        annualRate: annualRate.toString(),
      }));
      setIsManualSource(priceSrc === BigInt(1));
      console.log("Admin settings fetched successfully");
    } catch (e: unknown) {
      console.error("Admin settings fetch error:", e);
      handleError(e as Error, "Admin Settings");
      throw e;
    }
  }, [contract, isAdmin, handleError]);

  // Initialize chart
  const initChart = useCallback(() => {
    if (!chartRef.current) return;

    if (chartInstance.current) {
      chartInstance.current.destroy();
      chartInstance.current = null;
    }

    const ctx = chartRef.current.getContext("2d");
    if (!ctx) return;

    chartInstance.current = new Chart<"line", number[], string>(ctx, {
      type: "line",
      data: {
        labels: [],
        datasets: [
          {
            label: "Base Futures Curve",
            data: [],
            borderColor: "#1e40af",
            backgroundColor: "rgba(30, 64, 175, 0.1)",
            fill: true,
            tension: 0.2,
          },
          {
            label: "IV Adjusted Curve",
            data: [],
            borderColor: "#c026d3",
            backgroundColor: "rgba(192, 38, 211, 0.1)",
            borderDash: [5, 5],
            fill: true,
            tension: 0.2,
          },
        ],
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: { legend: { display: true } },
        scales: {
          x: {
            title: { display: true, text: "Maturity" },
            grid: { display: false },
          },
          y: { title: { display: true, text: "Price (IUSD)" } },
        },
      },
    });
  }, []);
const fetchChartData = useCallback(async () => {
  if (!contract || (oracleIsStale && !isManualSource) || !chartInstance.current) return;

  try {
    const labels: string[] = [];
    const base: number[] = [];
    const iv: number[] = [];
    const maturities: number[] = [];

    const nowSec = freezeNowSec();

    const today = new Date();
    for (let i = 0; i < 12; i++) {
      const d = new Date();
      d.setMonth(today.getMonth() + i, 15);
      d.setUTCHours(0, 0, 0, 0);
      labels.push(d.toLocaleString("default", { month: "short", year: "2-digit" }));
      maturities.push(Math.floor(d.getTime() / 1000));
    }

    const pricesWei = (await Promise.all(maturities.map((ts) => contract.priceFor(ts)))) as bigint[];
    const annualRate = Number((await contract.annualRateBPS()) as bigint) / 10000;
    const volatility = parseFloat(trade.spread.impliedVolatility || "0") / 100;

    pricesWei.forEach((wei, i) => {
      const s = parseFloat(ethers.formatUnits(wei, 18));
      base.push(s);
      if (trade.spread.ivEnabled) {
        const tYears = yearsToMaturity(maturities[i], nowSec);
        const prem = blackScholes("call", s, s, tYears, annualRate, volatility);
        iv.push(s + prem);
      }
    });

    chartInstance.current.data.labels = labels;
    chartInstance.current.data.datasets[0].data = base;
    chartInstance.current.data.datasets[1].data = iv;
    chartInstance.current.setDatasetVisibility(1, !!trade.spread.ivEnabled);
    chartInstance.current.update("none");
  } catch (e) {
    console.error("Chart data fetch error:", e);
    handleError(e as Error, "Chart Data");
  }
}, [
  contract,
  oracleIsStale,
  isManualSource,
  trade.spread.impliedVolatility,
  trade.spread.ivEnabled,
  handleError,
]);

  // Fetch chart data
  // const fetchChartData = useCallback(async () => {
  //   if (
  //     !contract ||
  //     (oracleIsStale && !isManualSource) ||
  //     !chartInstance.current
  //   ) {
  //     console.log("Skipping chart data fetch due to stale oracle");
  //     return;
  //   }

  //   try {
  //     const labels: string[] = [];
  //     const baseCurveData: number[] = [];
  //     const ivAdjustedCurveData: number[] = [];
  //     const today = new Date();
  //     const maturityTimestamps: number[] = [];

  //     const nowSec = Math.floor(Date.now() / 1000);

  //     for (let i = 0; i < 12; i++) {
  //       const futureDate = new Date();
  //       futureDate.setMonth(today.getMonth() + i, 15);
  //       futureDate.setUTCHours(0, 0, 0, 0);
  //       labels.push(
  //         futureDate.toLocaleString("default", {
  //           month: "short",
  //           year: "2-digit",
  //         })
  //       );
  //       maturityTimestamps.push(Math.floor(futureDate.getTime() / 1000));
  //     }

  //     const pricePromises = maturityTimestamps.map((ts) =>
  //       contract.priceFor(ts)
  //     );
  //     const basePricesWei = (await Promise.all(pricePromises)) as bigint[];

  //     const annualRate =
  //       Number((await contract.annualRateBPS()) as bigint) / 10000;
  //     const volatility = parseFloat(trade.spread.impliedVolatility) / 100;

  //     basePricesWei.forEach((priceWei: bigint, index: number) => {
  //       const basePrice = parseFloat(ethers.formatUnits(priceWei, 18));
  //       baseCurveData.push(basePrice);

  //       if (trade.spread.ivEnabled) {
  //         const maturityTs = maturityTimestamps[index];
  //         const timeToMaturityYears =
  //           (maturityTs - nowSec) / (365 * 24 * 60 * 60);
  //         const volatilityPremium = blackScholes(
  //           "call",
  //           basePrice,
  //           basePrice,
  //           timeToMaturityYears,
  //           annualRate,
  //           volatility
  //         );
  //         ivAdjustedCurveData.push(basePrice + volatilityPremium);
  //       }
  //     });

  //     chartInstance.current.data.labels = labels;
  //     chartInstance.current.data.datasets[0].data = baseCurveData;
  //     chartInstance.current.data.datasets[1].data = ivAdjustedCurveData;
  //     chartInstance.current.setDatasetVisibility(1, trade.spread.ivEnabled);
  //     chartInstance.current.update("none");
  //     console.log("Chart data fetched successfully");
  //   } catch (e: unknown) {
  //     console.error("Chart data fetch error:", e);
  //     handleError(e as Error, "Chart Data");
  //   }
  // }, [
  //   contract,
  //   oracleIsStale,
  //   isManualSource,
  //   trade.spread.impliedVolatility,
  //   trade.spread.ivEnabled,
  //   handleError,
  // ]);

  // Fetch portfolio data
  const fetchPortfolioData = useCallback(async () => {
    if (!contract || !userAddress || (oracleIsStale && !isManualSource)) {
      setPortfolio((prev) => ({ ...prev, positions: [] }));
      console.log("Skipping portfolio fetch due to stale oracle");
      return;
    }

    setPortfolio((prev) => ({ ...prev, isLoading: true }));
    try {
      const positionCount = (await contract.nextPositionIdForUser(
        userAddress
      )) as bigint;
      const count = Number(positionCount);
      const positionPromises: Promise<any>[] = []; // Raw from contract

      if (count > 0) {
        for (let i = 0; i < count; i++) {
          positionPromises.push(
            contract.getPositionById(userAddress, BigInt(i))
          );
        }
      }

      const allPositions = await Promise.all(positionPromises);
      const pricePromises: { [key: string]: Promise<bigint> } = {};
      const activePositions = allPositions.filter((p: any) => p.isActive);

      for (const p of activePositions) {
        for (const leg of p.legs) {
          const maturityStr = leg.maturity.toString();
          if (!pricePromises[maturityStr]) {
            pricePromises[maturityStr] = contract.priceFor(
              leg.maturity
            ) as Promise<bigint>;
          }
        }
      }

      const prices = await Promise.all(Object.values(pricePromises));
      const priceMap = Object.keys(pricePromises).reduce(
        (acc, maturity, index) => {
          acc[maturity] = prices[index];
          return acc;
        },
        {} as { [key: string]: bigint }
      );

      let totalPnl = BigInt(0);
      const positions: Position[] = activePositions
        .map((p: any) => {
          let posPnl = BigInt(0);
          let longAmount = BigInt(0);

          p.legs.forEach((leg: any) => {
            const currentPrice = priceMap[leg.maturity.toString()];
            const pnlForLeg =
              ((currentPrice - leg.entryPrice) * leg.amountIGLD) /
              ethers.parseUnits("1", 18);

            if (leg.isLong) {
              posPnl += pnlForLeg;
              longAmount += leg.amountIGLD;
            } else {
              posPnl -= pnlForLeg;
            }
          });

          totalPnl += posPnl;

          const derivedType =
            p.positionType === 0 || p.legs?.length === 1
              ? "Outright"
              : "Spread";

          return {
            positionId: p.positionId.toString(),
            positionType: derivedType,
            legs: p.legs.map((leg: any) => ({
              maturity: formatDateDDMMYYYY(
                new Date(Number(leg.maturity) * 1000)
              ),
              amountIGLD: parseFloat(ethers.formatUnits(leg.amountIGLD, 18)),
              isLong: leg.isLong,
              entryPrice: parseFloat(ethers.formatUnits(leg.entryPrice, 18)),
            })),
            longAmount: parseFloat(ethers.formatUnits(longAmount, 18)),
            stopLossPrice: parseFloat(ethers.formatUnits(p.stopLossPrice, 18)),
            takeProfitPrice: parseFloat(
              ethers.formatUnits(p.takeProfitPrice, 18)
            ),
            pnl: parseFloat(ethers.formatUnits(posPnl, 18)),
            currentPrice: (() => {
              const firstLeg = p.legs?.[0];
              if (!firstLeg) return null;
              const priceWei = priceMap[firstLeg.maturity.toString()];
              if (!priceWei) return null;
              return parseFloat(ethers.formatUnits(priceWei, 18));
            })(),
          };
        })
        .reverse();

      setPortfolio((prev) => ({
        ...prev,
        positions,
        pnl: parseFloat(ethers.formatUnits(totalPnl, 18)),
        activePositions: activePositions.length,
        isLoading: false,
      }));
    } catch (e: unknown) {
      console.error("Portfolio data fetch error:", e);
      handleError(e as Error, "Portfolio Data");
      setPortfolio((prev) => ({ ...prev, isLoading: false }));
    }
  }, [contract, userAddress, oracleIsStale, isManualSource, handleError]);

  // Reload data
  const reloadData = useCallback(async () => {
    if (!contract) return;

    try {
      setPortfolio((prev) => ({ ...prev, isLoading: true }));
      await fetchLiveSpotPrice();
      await fetchPortfolioData();
      if (activeTab === "trade" && chartInstance.current) {
        await fetchChartData();
      }
      if (isAdmin) {
        await fetchAdminSettings();
      }
    } catch (e: unknown) {
      console.error("Reload data error:", e);
      handleError(e as Error, "Data Reload");
    } finally {
      setPortfolio((prev) => ({ ...prev, isLoading: false }));
    }
  }, [
    contract,
    oracleIsStale,
    isManualSource,
    activeTab,
    isAdmin,
    fetchLiveSpotPrice,
    fetchPortfolioData,
    fetchChartData,
    fetchAdminSettings,
    handleError,
  ]);

// --- add near your imports (or top of file) ---
const erc20Abi = [
  "function balanceOf(address) view returns (uint256)",
  "function symbol() view returns (string)",
  "function decimals() view returns (uint8)"
];

// Debug helper: checks staticCall + payer balances for the chosen payout token
async function debugPayoutReadiness(
  contract: ethers.Contract,
  positionId: bigint,
  payoutToken: string,
  signer: ethers.Signer
) {
  const provider = signer.provider!;
  const core = (contract.target as string) || (await contract.getAddress?.());

  // A) Static simulation for this position & token
  try {
    await (contract as any).closePosition.staticCall(positionId, payoutToken);
    console.log("[debug] staticCall OK for", { positionId: positionId.toString(), payoutToken });
  } catch (e) {
    console.log("[debug] staticCall REVERT for", { positionId: positionId.toString(), payoutToken }, e);
  }

  // B) Balance checks at payer (core or vault)
  if (payoutToken !== ethers.ZeroAddress) {
    const IUSD = new ethers.Contract(payoutToken, erc20Abi, provider);
    const balCore = await IUSD.balanceOf(core);
    console.log("[debug] payout token @core:", balCore.toString());

    // If payout flows from a vault, log that too (guard in case no vault())
    const hasVault = typeof (contract as any).vault === "function";
    if (hasVault) {
      try {
        const vault = await (contract as any).vault();
        if (vault && vault !== ethers.ZeroAddress) {
          const balVault = await IUSD.balanceOf(vault);
          console.log("[debug] payout token @vault:", balVault.toString(), "vault:", vault);
        }
      } catch {/* no vault or not accessible */}
    }
  } else {
    // ETH payout (address(0)) — check core/vault native balance if needed
    const balCoreEth = await provider.getBalance(core);
    console.log("[debug] ETH @core:", balCoreEth.toString());
    try {
      const vault = await (contract as any).vault?.();
      if (vault) {
        const balVaultEth = await provider.getBalance(vault);
        console.log("[debug] ETH @vault:", balVaultEth.toString(), "vault:", vault);
      }
    } catch {
      
    }
  }
}


async function safeSendClosePosition(
  contract: ethers.Contract,
  positionId: bigint,
  payoutToken: string,
  signer: ethers.Signer
): Promise<TransactionReceipt> {
  const populated = await (contract as any).closePosition.populateTransaction(
    positionId,
    payoutToken
  );

  const from = await signer.getAddress();
  const to: string =
    (populated.to as string | undefined) ??
    ((contract as unknown as { target?: string }).target || "");
  const data: string = populated.data as string;
  const value: bigint =
    (populated.value as bigint | undefined) ?? BigInt(0);

  if (!to || !data) {
    throw new Error("closePosition populateTransaction failed: missing to/data");
  }

  console.log("[closePosition] populated:", {
    from,
    to,
    value: value.toString(),
    data,
  });

  try {
    console.log("[closePosition] running staticCall simulation...",positionId,payoutToken,value);
    const data = await (contract as any).closePosition.staticCall(positionId, payoutToken, {
      value,
    });
    console.log("[closePosition] staticCall: OK (no revert)", data);
  } catch (err: any) {
    console.log("[closePosition] staticCall REVERTED:", err);
    const liquidityMsg = interpretClosePositionLiquidityError(err);
    if (liquidityMsg) {
      throw new Error(liquidityMsg);
    }
    const msg = extractErrorMessage(err);
    throw new Error(`closePosition simulation failed: ${msg}`);
  }

  const unsignedTx: TransactionRequest = { to, data, value, from };
  const gas = await signer.estimateGas(unsignedTx);
  const gasLimit = (gas * BigInt(115)) / BigInt(100);

  console.log(
    "[closePosition] gas estimate:",
    gas.toString(),
    "buffered:",
    gasLimit.toString()
  );

  // 3) Send
  const tx = await signer.sendTransaction({ ...unsignedTx, gasLimit });

  console.log("[closePosition] submitted:", {
    hash: tx.hash,
    from: tx.from,
    to: tx.to,
  });

  // 4) Wait + guard null receipt (ethers v6 can return null)
  const receipt = await tx.wait();
  if (!receipt) {
    throw new Error("Transaction was dropped or replaced before confirmation.");
  }

  console.log("[closePosition] confirmed:", {
    txHash: receipt.hash,
    blockNumber: receipt.blockNumber,
    status: receipt.status,
  });

  return receipt;
}

  // Handle transaction lifecycle with latest reload logic
  const handleTx = useCallback(
    async (
      txPromise: Promise<ethers.TransactionResponse>,
      successMessage: string
    ) => {
      showToast("Sending transaction...", "info", 60000);
      try {
        const tx = await txPromise;
        showToast(
          "Transaction submitted, awaiting confirmation...",
          "info",
          60000
        );
        const receipt = await tx.wait();
        showToast(successMessage, "success");
        await reloadData();
        return receipt;
      } catch (e: unknown) {
        handleError(e as Error, "Transaction");
        throw e;
      }
    },
    [showToast, handleError, reloadData]
  );

  useEffect(() => {
    if (closeRefreshToken > 0) {
      fetchPortfolioData();
    }
  }, [closeRefreshToken, fetchPortfolioData]);

  // Check admin status
  const checkAdminStatus = useCallback(async () => {
    if (!contract || !userAddress) return;
    const contractAdmin = (await contract.admin()) as string;
    setIsAdmin(contractAdmin.toLowerCase() === userAddress.toLowerCase());
  }, [contract, userAddress]);

  // Get outright quote
  const getOutrightQuote = useCallback(async () => {
    const { maturity, amount, paymentToken, slippagePercent } = trade.outright;
    const allInputsValid = maturity && amount && parseFloat(amount) > 0;

    if (!allInputsValid || !contract) return;

    try {
      const sig = JSON.stringify({
        maturity,
        amount,
        paymentToken,
        slippagePercent,
      });
      if (lastOutrightSigRef.current === sig) return;
      lastOutrightSigRef.current = sig;
      const maturityTs = Math.floor(new Date(maturity).getTime() / 1000);
      const amount18 = ethers.parseUnits(amount, 18);

      const price = (await contract.priceFor(maturityTs)) as bigint;
      const principal = (amount18 * price) / ethers.parseUnits("1", 18);

      const feeBPS = (await contract.treasuryFeeBPS()) as bigint;
      const fee = (principal * feeBPS) / BigInt(10000);
      const totalCost = principal + fee;

      const slippageNum = parseFloat(trade.outright.slippagePercent);
      if (isNaN(slippageNum) || slippageNum < 0) return;

      const slippageBPS = BigInt(Math.round(slippageNum * 100));
      const maxCostWei =
        (totalCost * (BigInt(10000) + slippageBPS)) / BigInt(10000);
      setTrade((prev) => ({
        ...prev,
        outright: {
          ...prev.outright,
          maxCost: parseFloat(ethers.formatUnits(maxCostWei, 18)).toFixed(2),
        },
      }));
    } catch (e: unknown) {
      console.error("Outright quote failed:", e);
    }
  }, [contract, trade.outright]);

  // Debounced outright quote
  const debouncedGetOutrightQuote = useCallback(() => {
    const timer = setTimeout(() => getOutrightQuote(), 500);
    return () => clearTimeout(timer);
  }, [getOutrightQuote]);

  // Open outright position
  const handleExecuteOutrightTrade = useCallback(async () => {
    const { maturity, amount, paymentToken, maxCost, useSlTp, sl, tp } =
      trade.outright;
    if (!maturity || !amount || !maxCost) {
      showToast("Maturity, Amount, and Max Cost are required.", "error");
      return;
    }

    setIsOutrightTrading(true);
    try {
      const maturityTs = Math.floor(new Date(maturity).getTime() / 1000);
      const amount18 = ethers.parseUnits(amount, 18);
      let sl18 = ethers.parseUnits("0", 18);
      let tp18 = ethers.parseUnits("0", 18);
      if (useSlTp) {
        if (!contract) throw new Error("Contract not initialized");
        const entryPrice = (await contract.priceFor(maturityTs)) as bigint;
        sl18 = ethers.parseUnits(sl || "0", 18);
        tp18 = ethers.parseUnits(tp || "0", 18);

        if (sl18 > BigInt(0) && sl18 >= entryPrice) {
          showToast("Stop Loss must be BELOW entry price", "error");
          setIsOutrightTrading(false);
          return;
        }
        if (tp18 > BigInt(0) && tp18 <= entryPrice) {
          showToast("Take Profit must be ABOVE entry price", "error");
          setIsOutrightTrading(false);
          return;
        }
      }
      const maxCost18 = ethers.parseUnits(maxCost, 18);

      if (paymentToken === IUSD_ADDR) {
        const iusdContract = new Contract(IUSD_ADDR, ERC20_ALLOWANCE_ABI, signer);
        if (!contract) throw new Error("Contract not initialized");
        const price = (await contract.priceFor(maturityTs)) as bigint;
        const principal = (amount18 * price) / ethers.parseUnits("1", 18);
        const fee =
          (principal * ((await contract.treasuryFeeBPS()) as bigint)) /
          BigInt(10000);
        const totalCost = principal + fee;

        const allowance = (await iusdContract.allowance(
          userAddress,
          RFQ_ADDR
        )) as bigint;

        if (allowance < totalCost) {
          await handleTx(
            iusdContract.approve(RFQ_ADDR, MaxUint256),
            "IUSD Approval Successful"
          );
        }

        await handleTx(
          contract.openPositionWithIUSD(
            maturityTs,
            amount18,
            sl18,
            tp18,
            maxCost18
          ),
          "Outright position opened!"
        );
      } else {
        if (!contract) throw new Error("Contract not initialized");
        const price = (await contract.priceFor(maturityTs)) as bigint;
        const principal = (amount18 * price) / ethers.parseUnits("1", 18);
        const fee =
          (principal * ((await contract.treasuryFeeBPS()) as bigint)) /
          BigInt(10000);
        const overrides = { value: principal + fee };

        await handleTx(
          contract.openPositionWithEth(
            maturityTs,
            amount18,
            sl18,
            tp18,
            maxCost18,
            overrides
          ),
          "Outright position opened!"
        );
      }
    } catch (e: unknown) {
    } finally {
      setIsOutrightTrading(false);
    }
  }, [trade.outright, contract, signer, userAddress, handleTx, showToast]);

  const getSpreadQuote = useCallback(async () => {
    const { legs, ivEnabled, impliedVolatility, slippagePercent } =
      trade.spread;

    const validLegs =
      legs.length > 0 &&
      legs.every(
        (leg) =>
          leg.maturity &&
          leg.amountIGLD &&
          Number.isFinite(parseFloat(leg.amountIGLD)) &&
          parseFloat(leg.amountIGLD) > 0
      );

    if (!validLegs || !contract || (oracleIsStale && !isManualSource)) {
      setSpreadQuote({
        isCalculating: false,
        netCost: null,
        displayCost: "$0.00",
        error: validLegs ? "Invalid inputs or stale oracle." : null,
      });
      setTrade((prev) => ({
        ...prev,
        spread: { ...prev.spread, maxCost: "", minCredit: "" },
      }));
      return;
    }

    const sig = JSON.stringify({
      legs: legs.map((l) => ({
        m: l.maturity,
        a: l.amountIGLD,
        g: !!l.isLong,
      })),
      ivEnabled,
      impliedVolatility,
      slippagePercent,
    });

    if (lastSpreadSigRef.current === sig) return;
    lastSpreadSigRef.current = sig;

    const nowSec = Math.floor(Date.now() / 1000);
    quoteNowRef.current = nowSec;

    setSpreadQuote((prev) => ({ ...prev, isCalculating: true, error: null }));

    try {
      const maturityTs = legs.map((l) =>
        Math.floor(new Date(l.maturity).getTime() / 1000)
      );
      const amounts18 = legs.map((l) => ethers.parseUnits(l.amountIGLD, 18));
      const isLongFlags = legs.map((l) => !!l.isLong);

      const basePrices = (await Promise.all(
        maturityTs.map((ts) => contract.priceFor(ts))
      )) as bigint[];

      let prices: bigint[] = basePrices;
      if (ivEnabled) {
        const annualRate =
          Number((await contract.annualRateBPS()) as bigint) / 10000;
        const volatility = Math.max(0, parseFloat(impliedVolatility) / 100);

        prices = basePrices.map((wei, idx) => {
          const base = parseFloat(ethers.formatUnits(wei, 18));
          const tYears = Math.max(
            (maturityTs[idx] - nowSec) / (365 * 24 * 60 * 60),
            0
          );
          const premium = blackScholes(
            "call",
            base,
            base,
            tYears,
            annualRate,
            volatility
          );
          const adjusted = base + premium;
          return ethers.parseUnits(adjusted.toFixed(18), 18);
        });
      }

      const weiUnit = ethers.parseUnits("1", 18);
      let netPrincipalWei = BigInt(0);
      prices.forEach((priceWei, idx) => {
        const principalWei = (amounts18[idx] * priceWei) / weiUnit;
        netPrincipalWei += isLongFlags[idx] ? principalWei : -principalWei;
      });

      const absPrincipal = Math.abs(
        parseFloat(ethers.formatUnits(netPrincipalWei, 18))
      );
      const slipPct = parseFloat(slippagePercent) || 0;
      if (!Number.isFinite(slipPct) || slipPct < 0) {
        setSpreadQuote({
          isCalculating: false,
          netCost: null,
          displayCost: "$0.00",
          error: "Invalid slippage percentage.",
        });
        return;
      }

      const slipBps = BigInt(Math.round(Math.max(0, slipPct) * 100));

      if (netPrincipalWei > BigInt(0)) {
        const maxCostWei =
          (netPrincipalWei * (BigInt(10000) + slipBps)) / BigInt(10000);
        setSpreadQuote({
          isCalculating: false,
          netCost: netPrincipalWei,
          displayCost: `Debit: ${formatCurrency(absPrincipal)}`,
          error: null,
        });
        setTrade((prev) => ({
          ...prev,
          spread: {
            ...prev.spread,
            maxCost: parseFloat(ethers.formatUnits(maxCostWei, 18)).toFixed(2),
            minCredit: "",
          },
        }));
      } else if (netPrincipalWei < BigInt(0)) {
        const creditWei = -netPrincipalWei;
        const minCreditWei =
          (creditWei * (BigInt(10000) - slipBps)) / BigInt(10000);
        setSpreadQuote({
          isCalculating: false,
          netCost: netPrincipalWei,
          displayCost: `Credit: ${formatCurrency(absPrincipal)}`,
          error: null,
        });
        setTrade((prev) => ({
          ...prev,
          spread: {
            ...prev.spread,
            maxCost: "",
            minCredit: parseFloat(
              ethers.formatUnits(minCreditWei, 18)
            ).toFixed(2),
          },
        }));
      } else {
        setSpreadQuote({
          isCalculating: false,
          netCost: netPrincipalWei,
          displayCost: "$0.00",
          error: null,
        });
        setTrade((prev) => ({
          ...prev,
          spread: { ...prev.spread, maxCost: "0.00", minCredit: "0.00" },
        }));
      }

      if (activeTab === "trade" && chartInstance.current && !oracleIsStale) {
        await fetchChartData();
      }
    } catch (e) {
      handleError(e as Error, "Spread Quote");
      setSpreadQuote({
        isCalculating: false,
        netCost: null,
        displayCost: "$0.00",
        error: extractErrorMessage(e),
      });
      setTrade((prev) => ({
        ...prev,
        spread: { ...prev.spread, maxCost: "", minCredit: "" },
      }));
    }
  }, [
    activeTab,
    contract,
    fetchChartData,
    handleError,
    isManualSource,
    oracleIsStale,
    trade.spread,
  ]);
  // --- helper types & functions (keep once in the file) ---
// type SpreadLegCore = {
//   maturity: string | number | Date;
//   amountIGLD: string;                 // user input
//   isLong?: boolean | "true" | "false";
// };
// type LegacySide = { side?: "Buy" | "Sell" };
// type AnySpreadLeg = SpreadLegCore & LegacySide;

// const hasSide = (l: AnySpreadLeg): l is SpreadLegCore & Required<LegacySide> =>
//   typeof (l as any)?.side === "string";

// const normalizeIsLong = (leg: AnySpreadLeg): boolean => {
//   if (hasSide(leg)) return leg.side === "Buy";
//   if (typeof leg.isLong === "string") return leg.isLong === "true";
//   return !!leg.isLong;
// };

// const ETHER_1 = ethers.parseUnits("1", 18);

// // --- drop-in getSpreadQuote (fixes IV-OFF and IV-ON) ---
// const getSpreadQuote = useCallback(async () => {
//   // hard guard for TS and runtime
//   if (!contract) {
//     setSpreadQuote((p) => ({ ...p, isCalculating: false, error: null, netCost: null, displayCost: "$0.00" }));
//     setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost: "", minCredit: "" } }));
//     return;
//   }

//    const BIGINT_0 = BigInt(0);
//   const BIGINT_10000 = BigInt(10000);

//   const legs = (trade.spread.legs ?? []) as AnySpreadLeg[];
//   const valid = legs.length > 0 && legs.every((l) => {
//     const amt = parseFloat(String(l?.amountIGLD ?? ""));
//     return l?.maturity && Number.isFinite(amt) && amt > 0;
//   });

//   if (!valid) {
//     setSpreadQuote((p) => ({ ...p, isCalculating: false, error: null, netCost: null, displayCost: "$0.00" }));
//     setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost: "", minCredit: "" } }));
//     return;
//   }

//   setSpreadQuote((p) => ({ ...p, isCalculating: true, error: null }));

//   // Freeze "now" only once and only used when IV is ON
//   const includeIV = !!trade.spread.ivEnabled;
//   const nowForQuoteSec = includeIV ? (Date.now() / 1000) : undefined;

//   try {
//     let netCostWei = BIGINT_0;

//     // Only fetch rates when IV is ON (prevents NaNs & wasted RPCs)
//     let annualRate = 0;
//     let volatility = 0;
//     if (includeIV) {
//       const rateBps = (await contract.annualRateBPS()) as bigint;  // e.g., 500 -> 5%
//       annualRate = Number(rateBps) / 10000;
//       volatility = parseFloat(String(trade.spread.impliedVolatility ?? "0")) / 100; // "20" -> 0.20
//       if (!Number.isFinite(annualRate)) annualRate = 0;
//       if (!Number.isFinite(volatility)) volatility = 0;
//     }

//     for (const leg of legs) {
//       const matTs = Math.floor(new Date(leg.maturity as any).getTime() / 1000);
//       const amt18 = ethers.parseUnits(String(leg.amountIGLD), 18);
//       let priceWei = (await contract.priceFor(matTs)) as bigint;

//       if (includeIV) {
//         const base = parseFloat(ethers.formatUnits(priceWei, 18));
//         const tYears = Math.max(0, (matTs - (nowForQuoteSec as number)) / (365 * 24 * 60 * 60));
//         const premium = (annualRate >= 0 && Number.isFinite(base) && Number.isFinite(tYears) && Number.isFinite(volatility))
//           ? blackScholes("call", base, base, tYears, annualRate, volatility)
//           : 0;
//         const simulated = base + premium;
//         priceWei = ethers.parseUnits(simulated.toFixed(18), 18);
//       }

//       const legWei = (priceWei * amt18) / ETHER_1;

//       const isLong = normalizeIsLong(leg);
//       netCostWei = isLong ? netCostWei + legWei : netCostWei - legWei;
//     }

//     const absFloat = parseFloat(ethers.formatUnits(netCostWei < BIGINT_0 ? -netCostWei : netCostWei, 18));
//     const formattedCost = Math.abs(absFloat);

//     let slippagePercent = parseFloat(String(trade.spread.slippagePercent ?? "0"));
//     if (!Number.isFinite(slippagePercent) || slippagePercent < 0) slippagePercent = 0;
//     const slippageBPS = BigInt(Math.round(slippagePercent * 100)); // % → BPS

//     if (netCostWei > BIGINT_0) {
//       setSpreadQuote((p) => ({ ...p, netCost: netCostWei, displayCost: `Debit: $${formattedCost.toFixed(2)}`, error: null }));
//       const maxCostWei = (netCostWei * (BIGINT_10000 + slippageBPS)) / BIGINT_10000;
//       const maxCost = parseFloat(ethers.formatUnits(maxCostWei, 18)).toFixed(2);
//       setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost, minCredit: "" } }));
//     } else if (netCostWei < BIGINT_0) {
//       // Credit path
//       setSpreadQuote((p) => ({ ...p, netCost: netCostWei, displayCost: `Credit: $${formattedCost.toFixed(2)}`, error: null }));
//       const creditWei = -netCostWei;
//       const minCreditWei = (creditWei * (BIGINT_10000 - slippageBPS)) / BIGINT_10000;
//       const minCredit = parseFloat(ethers.formatUnits(minCreditWei, 18)).toFixed(2);
//       setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost: "", minCredit } }));
//     } else {
//       // Zero
//       setSpreadQuote((p) => ({ ...p, netCost: netCostWei, displayCost: "$0.00", error: null }));
//       setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost: "", minCredit: "" } }));
//     }
//   } catch (e) {
//     console.error("Spread quote failed:", e);
//     setSpreadQuote((p) => ({
//       ...p,
//       isCalculating: false,
//       error: "Could not calculate price. Invalid inputs.",
//       netCost: null,
//       displayCost: "$0.00",
//     }));
//     setTrade((prev) => ({ ...prev, spread: { ...prev.spread, maxCost: "", minCredit: "" } }));
//   } finally {
//     setSpreadQuote((p) => ({ ...p, isCalculating: false }));
//   }
// }, [contract, trade.spread]);



  // Debounced spread quote
  const debouncedGetSpreadQuote = useCallback(() => {
    const timer = setTimeout(() => getSpreadQuote(), 500);
    return () => clearTimeout(timer);
  }, [getSpreadQuote]);

  const memoizedLegs = useMemo(() => trade.spread.legs, [trade.spread.legs]);
  // Open spread position
  const handleExecuteSpreadTrade = useCallback(async () => {
    const { legs, paymentToken, slippagePercent } = trade.spread;
    const allInputsValid = legs.every(
      (leg) => leg.maturity && leg.amountIGLD && parseFloat(leg.amountIGLD) > 0
    );

    if (!allInputsValid) {
      showToast(
        "All spread legs must have a valid maturity and amount.",
        "error"
      );
      return;
    }

    let netIGLDAmount = BigInt(0);
    legs.forEach((leg) => {
      const amount = ethers.parseUnits(leg.amountIGLD || "0", 18);
      if (leg.isLong) {
        netIGLDAmount += amount;
      } else {
        netIGLDAmount -= amount;
      }
    });

    if (netIGLDAmount < BigInt(0)) {
      showToast(
        "Error: Net short positions are not supported. Total long amount must be >= total short amount.",
        "error",
        6000
      );
      return;
    }

    setIsSpreadTrading(true);
    try {
      const inputLegs = legs.map((leg) => ({
        maturity: Math.floor(new Date(leg.maturity).getTime() / 1000),
        amountIGLD: ethers.parseUnits(leg.amountIGLD, 18),
        isLong: leg.isLong,
      }));

      let actualNetCostWei = BigInt(0);
      for (const leg of inputLegs) {
        if (!contract) throw new Error("Contract not initialized");
        const priceWei = (await contract.priceFor(leg.maturity)) as bigint;
        const legValue =
          (priceWei * leg.amountIGLD) / ethers.parseUnits("1", 18);
        if (leg.isLong) {
          actualNetCostWei += legValue;
        } else {
          actualNetCostWei -= legValue;
        }
      }

      let finalMaxCost18 = BigInt(0);
      let finalMinCredit18 = BigInt(0);
      let overrides: { value?: bigint } = {};

      const slippageBPS = BigInt(Math.round(parseFloat(slippagePercent) * 100));
      if (isNaN(Number(slippageBPS)) || Number(slippageBPS) < 0) {
        showToast("Invalid slippage percentage.", "error");
        return;
      }

      if (!contract) throw new Error("Contract not initialized");
      const feeBPS = (await contract.treasuryFeeBPS()) as bigint;

      if (actualNetCostWei > BigInt(0)) {
        const fee = (actualNetCostWei * feeBPS) / BigInt(10000);
        const totalCost = actualNetCostWei + fee;
        finalMaxCost18 =
          (totalCost * (BigInt(10000) + slippageBPS)) / BigInt(10000);

        if (paymentToken === IUSD_ADDR) {
          const iusdContract = new Contract(
            IUSD_ADDR,
            ERC20_ALLOWANCE_ABI,
            signer
          );
          const allowance = (await iusdContract.allowance(
            userAddress,
            RFQ_ADDR
          )) as bigint;
          if (allowance < totalCost) {
            await handleTx(
              iusdContract.approve(RFQ_ADDR, MaxUint256),
              "IUSD Approval Successful"
            );
          }
        } else {
          overrides = { value: totalCost };
        }
      } else if (actualNetCostWei < BigInt(0)) {
        const totalCredit = -actualNetCostWei;
        finalMinCredit18 =
          (totalCredit * (BigInt(10000) - slippageBPS)) / BigInt(10000);
      }

      await handleTx(
        contract.openSpreadPosition(
          inputLegs,
          paymentToken,
          finalMaxCost18,
          finalMinCredit18,
          overrides
        ),
        "Spread position opened!"
      );
    } catch (e: unknown) {
      // Error handled in handleTx
    } finally {
      setIsSpreadTrading(false);
    }
  }, [trade.spread, contract, signer, userAddress, handleTx, showToast]);

  // Close position
  const closePositionOnChain = useCallback(
    async (position: Position) => {
      if (!contract) throw new Error("Contract not initialized");
      try {
        const posData = (await contract.getPositionById(
          userAddress,
          BigInt(position.positionId)
        )) as any;
        let totalLongAmountToBurn = BigInt(0);
        let pnlValue = BigInt(0);

        const pricePromises = posData.legs.map((leg: any) =>
          contract.priceFor(leg.maturity)
        );
        const currentPrices = (await Promise.all(pricePromises)) as bigint[];

        posData.legs.forEach((leg: any, index: number) => {
          const currentPrice = currentPrices[index];
          const pnlForLeg =
            ((currentPrice - leg.entryPrice) * leg.amountIGLD) /
            ethers.parseUnits("1", 18);

          if (leg.isLong) {
            pnlValue += pnlForLeg;
            totalLongAmountToBurn += leg.amountIGLD;
          } else {
            pnlValue -= pnlForLeg;
          }
        });

        if (pnlValue < BigInt(0)) {
          const lossAmount = -pnlValue;
          const iusdContract = new Contract(
            IUSD_ADDR,
            ERC20_ALLOWANCE_ABI,
            signer
          );
          const allowance = (await iusdContract.allowance(
            userAddress,
            RFQ_ADDR
          )) as bigint;

          if (allowance < lossAmount) {
            await handleTx(
              iusdContract.approve(RFQ_ADDR, MaxUint256),
              "IUSD Approval for Loss Successful"
            );
          }
        }

        if (totalLongAmountToBurn > BigInt(0)) {
          const igldContract = new Contract(
            IGLD_ADDR,
            ERC20_ALLOWANCE_ABI,
            signer
          );
          const allowance = (await igldContract.allowance(
            userAddress,
            RFQ_ADDR
          )) as bigint;

          if (allowance < totalLongAmountToBurn) {
            await handleTx(
              igldContract.approve(RFQ_ADDR, MaxUint256),
              "IGLD Approval Successful"
            );
          }
        }
        const wait = await debugPayoutReadiness(
        contract,
        BigInt(position.positionId),
        IUSD_ADDR, // or whichever payout token you plan to use
        signer!
      );
console.log("Payout readiness check complete:", wait);

        // await handleTx(
        //   contract.closePosition(
        //     BigInt(position.positionId),
        //     IUSD_ADDR
        //   ),
        //   `Position #${position.positionId} closed.`
        // );
        console.log("Closing position on-chain:", position.positionId);
          const receipt = await safeSendClosePosition(
    contract,
    BigInt(position.positionId), // no literal: BigInt(...)
    IUSD_ADDR,          // or your chosen payout token
    signer!                      // safe if you just guarded above
  );

  console.log("Close position transaction receipt:", receipt);
  showToast(`Position #${position.positionId} closed.`, "success");
  await reloadData();
        console.log("Position closed on-chain.");
      } catch (e: unknown) {
        if (isUserRejectedError(e)) {
          throw e;
        }
        handleError(e as Error, "Close Position");
        throw e;
      }
    },
    [contract, userAddress, signer, handleTx, isUserRejectedError, handleError]
  );

  // useEffect(() => {
  //   if (!portfolio.positions.length || isCloseProcessing) return;

  //   const now = Date.now();
  //   const shouldAutoClose = (position: Position) => {
  //     if (position.positionType.toLowerCase() !== "outright") return false;
  //     if (!Number.isFinite(position.currentPrice ?? NaN)) return false;
  //     const price = position.currentPrice as number;
  //     const sl = position.stopLossPrice;
  //     const tp = position.takeProfitPrice;
  //     return (sl > 0 && price <= sl) || (tp > 0 && price >= tp);
  //   };

  //   const candidates = portfolio.positions.filter(shouldAutoClose);
  //   if (!candidates.length) return;

  //   for (const position of candidates) {
  //     const id = position.positionId;
  //     if (autoCloseInFlightRef.current.has(id)) continue;
  //     const lastAttempt = autoCloseLastAttemptRef.current[id] || 0;
  //     if (now - lastAttempt < 60000) continue;

  //     autoCloseInFlightRef.current.add(id);
  //     autoCloseLastAttemptRef.current[id] = now;
  //     void closePositionOnChain(position)
  //       .catch(() => {
  //         // Errors are already surfaced by closePositionOnChain.
  //       })
  //       .finally(() => {
  //         autoCloseInFlightRef.current.delete(id);
  //       });
  //   }
  // }, [portfolio.positions, isCloseProcessing, closePositionOnChain]);

  const handleRequestClosePosition = useCallback((position: Position) => {
    setPendingClosePosition(position);
  }, []);

  const handleCancelClosePosition = useCallback(() => {
    if (isCloseProcessing) return;
    setPendingClosePosition(null);
  }, [isCloseProcessing]);

  const handleConfirmClosePosition = useCallback(async () => {
    if (!pendingClosePosition) return;
    setIsCloseProcessing(true);
    try {
      await closePositionOnChain(pendingClosePosition);
      setPendingClosePosition(null);
      setCloseRefreshToken((prev) => prev + 1);
    } catch (err) {
      if (isUserRejectedError(err)) {
        setPendingClosePosition(null);
        showToast(getUserRejectedMessage(err), "warning");
      }
    } finally {
      setIsCloseProcessing(false);
    }
  }, [
    pendingClosePosition,
    closePositionOnChain,
    isUserRejectedError,
    showToast,
    getUserRejectedMessage,
  ]);

  // Admin functions
  const handleSetManualSpotPrice = useCallback(async () => {
    const price = admin.manualSpotPrice;
    if (!price) {
      showToast("Manual price cannot be empty.", "error");
      return;
    }
    if (!contract) return;
    await handleTx(
      contract.setManualPrice(ethers.parseUnits(price, 18)),
      "Manual price set."
    );
  }, [admin.manualSpotPrice, contract, handleTx, showToast]);

  const handleAddKeeper = useCallback(async () => {
    if (!admin.keeperAddress || !ethers.isAddress(admin.keeperAddress)) {
      showToast("Keeper address is invalid.", "error");
      return;
    }
    if (!contract) return;
    await handleTx(
      contract.setKeeper(admin.keeperAddress, true),
      "Keeper added."
    );
  }, [admin.keeperAddress, contract, handleTx, showToast]);

  const handleRemoveKeeper = useCallback(async () => {
    if (!admin.keeperAddress || !ethers.isAddress(admin.keeperAddress)) {
      showToast("Keeper address is invalid.", "error");
      return;
    }
    if (!contract) return;
    await handleTx(
      contract.setKeeper(admin.keeperAddress, false),
      "Keeper removed."
    );
  }, [admin.keeperAddress, contract, handleTx, showToast]);

  const handleSetAnnualRateBPS = useCallback(async () => {
    if (!admin.annualRate) {
      showToast("Annual Rate BPS cannot be empty.", "error");
      return;
    }
    if (!contract) return;
    await handleTx(
      contract.setAnnualRateBPS(BigInt(admin.annualRate)),
      "Annual rate updated."
    );
  }, [admin.annualRate, contract, handleTx, showToast]);

  const handleSetOracleMaxAge = useCallback(async () => {
    if (!admin.oracleMaxAge) {
      showToast("Oracle Max Age cannot be empty.", "error");
      return;
    }
    if (!contract) return;
    await handleTx(
      contract.setOracleMaxAge(BigInt(admin.oracleMaxAge)),
      "Oracle max age updated."
    );
  }, [admin.oracleMaxAge, contract, handleTx, showToast]);

  // Set price source (optimistic UI update)
  const handleSetPriceSource = useCallback(
    async (source: "Manual" | "Chainlink") => {
      if (!contract) return;
      await handleTx(
        contract.setPriceSource(source === "Manual" ? 1 : 0),
        "Price source updated."
      );
      setIsManualSource(source === "Manual");
    },
    [contract, handleTx]
  );

  // Analyze carry
  const handleAnalyzeCarry = useCallback(async () => {
    if (!analysis.startDate || !analysis.endDate) {
      showToast("Please select both a start and end date.", "error");
      return;
    }
    if (oracleIsStale && !isManualSource) return;

    const d1 = new Date(analysis.startDate);
    const d2 = new Date(analysis.endDate);

    if (d1 >= d2) {
      showToast("Start date must be before end date.", "error");
      return;
    }

    setAnalysis((prev) => ({ ...prev, results: "loading" }));
    try {
      const ts1 = Math.floor(d1.getTime() / 1000);
      const ts2 = Math.floor(d2.getTime() / 1000);
      if (!contract) throw new Error("Contract not initialized");
      const [price1, price2] = await Promise.all([
        contract.priceFor(ts1) as Promise<bigint>,
        contract.priceFor(ts2) as Promise<bigint>,
      ]);

      const p1 = parseFloat(ethers.formatUnits(price1, 18));
      const p2 = parseFloat(ethers.formatUnits(price2, 18));
      const priceSpread = p2 - p1;
      const basisPoints = p1 > 0 ? (priceSpread / p1) * 10000 : 0;
      const daysBetween = (d2.getTime() - d1.getTime()) / (1000 * 60 * 60 * 24);
      const annualizedYield =
        p1 > 0 && daysBetween > 0
          ? (priceSpread / p1) * (365 / daysBetween) * 100
          : 0;

      setAnalysis((prev) => ({
        ...prev,
        results: {
          priceSpread,
          basisPoints,
          annualizedYield,
          startEndPrice: `${formatCurrency(p1)} / ${formatCurrency(p2)}`,
        },
      }));
    } catch (e: unknown) {
      console.error("Carry analysis error:", e);
      handleError(e as Error, "Analysis");
      setAnalysis((prev) => ({ ...prev, results: null }));
    }
  }, [
    analysis.startDate,
    analysis.endDate,
    oracleIsStale,
    isManualSource,
    contract,
    showToast,
    handleError,
  ]);

  // Effects for initialization and updates
  useEffect(() => {
    const eth = (window as any).ethereum;
    if (eth) {
      connectWallet();
      (eth as any).on("accountsChanged", () => window.location.reload());
      (eth as any).on("chainChanged", () => window.location.reload());
      return () => {
        const ethCleanup = (window as any).ethereum;
        if (ethCleanup) {
          (ethCleanup as any).removeListener("accountsChanged", () =>
            window.location.reload()
          );
          (ethCleanup as any).removeListener("chainChanged", () =>
            window.location.reload()
          );
        }
      };
    } else {
      showToast("Please install a web3 wallet like MetaMask.", "error", 5000);
    }
  }, [connectWallet, showToast]);

  useEffect(() => {
    if (contract && userAddress) {
      checkAdminStatus();
    }
  }, [contract, userAddress, checkAdminStatus]);

  useEffect(() => {
    if (activeTab === "trade" && !chartInstance.current) {
      initChart();
      fetchChartData();
    }
    if (activeTab === "admin" && isAdmin) {
      fetchAdminSettings();
    }
  }, [activeTab, isAdmin, initChart, fetchChartData, fetchAdminSettings]);

  useEffect(() => {
    if (activeTab === "trade") {
      // Initialize chart if it doesn't exist or canvas is valid
      if (!chartInstance.current && chartRef.current) {
        initChart();
      }
      // Fetch chart data when trade tab is active
      if (chartInstance.current && !oracleIsStale) {
        fetchChartData();
      }
    } else {
      // Destroy chart when leaving trade tab
      if (chartInstance.current) {
        chartInstance.current.destroy();
        chartInstance.current = null;
      }
    }

    // Cleanup on unmount
    return () => {
      if (chartInstance.current) {
        chartInstance.current.destroy();
        chartInstance.current = null;
      }
    };
  }, [activeTab, initChart, fetchChartData, oracleIsStale]);

  // Update chart data when implied volatility or ivEnabled changes
  useEffect(() => {
    if (activeTab === "trade" && chartInstance.current && !oracleIsStale) {
      fetchChartData();
    }
  }, [
    trade.spread.impliedVolatility,
    trade.spread.ivEnabled,
    fetchChartData,
    activeTab,
    oracleIsStale,
  ]);

  useEffect(() => {
    debouncedGetOutrightQuote();
  }, [
    trade.outright.maturity,
    trade.outright.amount,
    trade.outright.slippagePercent,
    debouncedGetOutrightQuote,
  ]);

  useEffect(() => {
    debouncedGetSpreadQuote();
  }, [
    trade.spread.legs,
    trade.spread.impliedVolatility,
    trade.spread.ivEnabled,
    trade.spread.slippagePercent,
    debouncedGetSpreadQuote,
  ]);

  return (
    <Box sx={{ p: { xs: 2, md: 4 } }}>
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 4,
        }}
      >
        <Typography
          variant="h4"
          sx={{ fontWeight: "bold", color: theme.palette.text.primary }}
        >
          IntelliGold Futures OTC RFQ
        </Typography>
        <Box sx={{ display: "flex", alignItems: "center", gap: 2 }}>
          <Chip
            label={status.message}
            color={status.type === "connected" ? "success" : "default"}
            icon={status.type === "connected" ? <TaskAltIcon /> : undefined}
            sx={{ fontWeight: "medium" }}
            aria-label={`Connection status: ${status.message}`}
          />
          {status.type !== "connected" && (
            <Button
              variant="outlined"
              onClick={connectWallet}
              disabled={isLoading}
              aria-label="Connect wallet"
              sx={{
                bgcolor: "#0b3d91",
                textTransform: "none",
                fontSize: 14,
                color: "#FFFFFF",
                "&:hover": { bgcolor: "#083475" },
              }}
            >
              {isLoading ? (
                <CircularProgress size={24} color="inherit" />
              ) : (
                "Connect Wallet"
              )}
            </Button>
          )}
          {status.type === "connected" && (
            <Button
              variant="outlined"
              startIcon={<LogoutIcon />}
              onClick={disconnectWallet}
              aria-label="Disconnect wallet"
              sx={{
                bgcolor: "#0b3d91",
                textTransform: "none",
                fontSize: 14,
                color: "#FFFFFF",
                "&:hover": { bgcolor: "#083475" },
              }}
            >
              Disconnect
            </Button>
          )}
        </Box>
      </Box>

      {oracleIsStale && !isManualSource && (
        <Paper
          sx={{
            p: 2,
            mb: 4,
            bgcolor: theme.palette.error.light,
            color: theme.palette.error.contrastText,
            border: "1px solid #D4DBE3",
            boxShadow: "none",
          }}
          role="alert"
          aria-label="Oracle warning"
        >
          <Typography variant="body1" sx={{ fontWeight: "medium" }}>
            Oracle Warning
          </Typography>
          <Typography variant="body2">
            The Chainlink price feed is stale. Trading and analysis are disabled
            until the oracle updates or an admin switches to a manual price
            source in the Admin Panel.
          </Typography>
        </Paper>
      )}

      {userAddress && (
        <main>
          <Tabs
            value={activeTab}
            onChange={(_, newValue: "portfolio" | "trade" | "admin") =>
              setActiveTab(newValue)
            }
            sx={{
               mb: 4 , 
               "& .MuiTab-root": {
                    textTransform: "none",  
                    fontSize: 16,
                  },
                 }}
            aria-label="Navigation tabs"
          >
            <Tab
              label="Portfolio"
              value="portfolio"
              aria-label="Portfolio tab"
            />
            <Tab label="Trade" value="trade" aria-label="Trade tab" />
            {isAdmin && (
              <Tab
                label="Admin Panel"
                value="admin"
                aria-label="Admin panel tab"
              />
            )}
          </Tabs>

          {activeTab === "portfolio" && (
            <Box>
              <Box
                sx={{
                  display: "grid",
                  gap: 2,
                  gridTemplateColumns: {
                    xs: "1fr",
                    sm: "1fr 1fr",
                    md: "1fr 1fr 1fr",
                  },
                  mb: 4,
                }}
              >
                <Paper
                  elevation={0}
                  sx={{
                    p: 2,
                    textAlign: "center",
                    bgcolor: "#FFFFFF",
                    color: theme.palette.text.primary,
                    border: "1px solid #E0E0E0",
                    boxShadow: "none",
                  }}
                >
                  <Typography variant="body2" sx={{ fontWeight: "medium" }}>
                    Unrealized P/L
                  </Typography>
                  <Typography
                    variant="h5"
                    sx={{
                      fontWeight: "bold",
                      color: portfolio.isLoading
                        ? theme.palette.text.primary
                        : getPnlColor(portfolio.pnl),
                    }}
                  >
                    {portfolio.isLoading
                      ? "Loading..."
                      : formatPnlWithSign(portfolio.pnl)}
                  </Typography>
                </Paper>
                <Paper
                  elevation={0}
                  sx={{
                    p: 2,
                    textAlign: "center",
                    bgcolor: "#FFFFFF",
                    color: theme.palette.text.primary,
                    border: "1px solid #E0E0E0",
                    boxShadow: "none",
                  }}
                >
                  <Typography variant="body2" sx={{ fontWeight: "medium" }}>
                    Active Positions
                  </Typography>
                  <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                    {portfolio.isLoading
                      ? "Loading..."
                      : portfolio.activePositions}
                  </Typography>
                </Paper>
                <Paper
                  elevation={0}
                  sx={{
                    p: 2,
                    textAlign: "center",
                    bgcolor: "#FFFFFF",
                    color: theme.palette.text.primary,
                    border: "1px solid #E0E0E0",
                    boxShadow: "none",
                  }}
                >
                  <Typography variant="body2" sx={{ fontWeight: "medium" }}>
                    Spot Gold Price
                  </Typography>
                  <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                    {portfolio.isLoading
                      ? "Loading..."
                      : formatCurrency(portfolio.spotPrice, 2)}
                  </Typography>
                </Paper>
              </Box>

              <TableContainer
                component={Paper}
                sx={{ border: "1px solid #D4DBE3", boxShadow: "none" }}
              >
                <Table aria-label="Portfolio positions table">
                  <TableHead>
                    <TableRow>
                      <TableCell>ID</TableCell>
                      <TableCell>Type</TableCell>
                      <TableCell>Legs</TableCell>
                      <TableCell>Long Amount</TableCell>
                      <TableCell>Current P/L</TableCell>
                      <TableCell>Action</TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {portfolio.isLoading ? (
                      <TableRow>
                        <TableCell colSpan={6} align="center">
                          Loading portfolio...
                        </TableCell>
                      </TableRow>
                    ) : oracleIsStale && !isManualSource ? (
                      <TableRow>
                        <TableCell colSpan={6} align="center">
                          Portfolio data unavailable due to stale oracle. If you
                          are admin, switch to manual source in Admin Panel.
                        </TableCell>
                      </TableRow>
                    ) : portfolio.positions.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={6} align="center">
                          No active positions.
                        </TableCell>
                      </TableRow>
                    ) : (
                      portfolio.positions.map((position) => (
                        <TableRow key={position.positionId}>
                          <TableCell>{position.positionId}</TableCell>
                          <TableCell>
                            {renderPositionTypeChip(position.positionType)}
                          </TableCell>
                          <TableCell>
                            <Box
                              sx={{
                                display: "flex",
                                flexDirection: "column",
                                gap: 0.5,
                              }}
                            >
                              {position.legs.map((leg, idx) => (
                                <Typography
                                  key={idx}
                                  variant="body2"
                                  sx={{
                                    fontWeight: 600,
                                    color: leg.isLong ? "#10B981" : "#EF4444",
                                  }}
                                >
                                  <Typography
                                    component="span"
                                    variant="body2"
                                    sx={{
                                      color: leg.isLong ? "#10B981" : "#EF4444",
                                      fontWeight: 700,
                                      mr: 0.5,
                                    }}
                                  >
                                    {leg.isLong ? "Long" : "Short"}
                                  </Typography>
                                  <Typography
                                    component="span"
                                    variant="body2"
                                    sx={{ fontWeight: 500, color: "#374151" }}
                                  >
                                    {`${leg.amountIGLD.toFixed(2)} @ ${
                                      leg.maturity
                                    }`}
                                  </Typography>{" "}
                                  <Typography
                                    component="span"
                                    variant="body2"
                                    sx={{
                                      color: "text.secondary",
                                      fontWeight: 500,
                                    }}
                                  >
                                    (Entry: {formatCurrency(leg.entryPrice)})
                                  </Typography>
                                </Typography>
                              ))}
                            </Box>
                          </TableCell>
                          <TableCell>
                            {position.legs
                              .reduce(
                                (sum, leg) =>
                                  sum +
                                  (leg.isLong
                                    ? leg.amountIGLD
                                    : -leg.amountIGLD),
                                0
                              )
                              .toFixed(2)}{" "}
                            IGLD
                          </TableCell>
                          <TableCell
                            sx={{
                              color:
                                position.pnl > 0
                                  ? "#10B981"
                                  : position.pnl < 0
                                  ? "#EF4444"
                                  : "#6B7280",
                              fontWeight: 600,
                            }}
                          >
                            {formatPnlWithSign(position.pnl)}
                          </TableCell>
                          <TableCell>
                            <Button
                              variant="contained"
                              size="small"
                              disableElevation
                              onClick={() =>
                                handleRequestClosePosition(position)
                              }
                              disabled={
                                (oracleIsStale && !isManualSource) ||
                                (isCloseProcessing &&
                                  pendingClosePosition?.positionId ===
                                    position.positionId)
                              }
                              aria-label={`Close position ${position.positionId}`}
                              sx={{
                                borderRadius: "8px",
                                px: 2.5,
                                fontWeight: 600,
                                textTransform: "none",
                                color: "#B91C1C",
                                backgroundImage:
                                  "linear-gradient(180deg, rgba(254, 236, 236, 0.96) 0%, rgba(252, 219, 219, 0.96) 100%)",
                                border: "1px solid rgba(248, 113, 113, 0.35)",
                                boxShadow:
                                  "0px 3px 8px rgba(190, 49, 68, 0.16)",
                                "&:hover": {
                                  backgroundImage:
                                    "linear-gradient(180deg, rgba(253, 222, 222, 1) 0%, rgba(251, 209, 209, 1) 100%)",
                                  borderColor: "rgba(220, 38, 38, 0.4)",
                                  boxShadow:
                                    "0px 4px 12px rgba(190, 49, 68, 0.18)",
                                },
                                "&:active": {
                                  backgroundImage:
                                    "linear-gradient(180deg, rgba(252, 210, 210, 1) 0%, rgba(249, 196, 196, 1) 100%)",
                                  boxShadow:
                                    "0px 2px 6px rgba(190, 49, 68, 0.2)",
                                },
                                "&:focus-visible": {
                                  outline: "2px solid rgba(244, 63, 94, 0.45)",
                                  outlineOffset: "2px",
                                },
                              }}
                            >
                              {isCloseProcessing &&
                              pendingClosePosition?.positionId ===
                                position.positionId
                                ? "Closing..."
                                : "Close"}
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </TableContainer>
            </Box>
          )}

          {activeTab === "trade" && (
            <Box>
              <Typography
                variant="h5"
                sx={{
                  fontWeight: "bold",
                  mb: 3,
                  color: theme.palette.text.primary,
                }}
              >
                Gold Futures Curve
              </Typography>
              <Paper
                sx={{
                  p: 4,
                  mb: 4,
                  height: 400,
                  border: "1px solid #D4DBE3",
                  boxShadow: "none",
                }}
              >
                {oracleIsStale && !isManualSource ? (
                  <Typography variant="body1" color="error" align="center">
                    Graph data unavailable due to stale oracle. If you are
                    admin, switch to manual source in Admin Panel.
                  </Typography>
                ) : (
                  <canvas
                    ref={chartRef}
                    aria-label="Gold futures curve chart"
                  />
                )}
              </Paper>

              <Typography
                variant="h5"
                sx={{
                  fontWeight: "bold",
                  mb: 3,
                  color: theme.palette.text.primary,
                }}
              >
                Interactive Carry Analysis
              </Typography>
              <Paper
                sx={{
                  p: 4,
                  mb: 4,
                  border: "1px solid #D4DBE3",
                  boxShadow: "none",
                }}
              >
                <Box
                  sx={{
                    display: "flex",
                    flexDirection: { xs: "column", sm: "row" },
                    gap: 2,
                    mb: 4,
                  }}
                >
                  <TextField
                    label="Start Date"
                    type="date"
                    value={analysis.startDate}
                    onChange={(e: ChangeEvent<HTMLInputElement>) =>
                      setAnalysis({ ...analysis, startDate: e.target.value })
                    }
                    InputLabelProps={{ shrink: true }}
                    sx={{ flex: 1 }}
                    aria-label="Start date for carry analysis"
                  />
                  <TextField
                    label="End Date"
                    type="date"
                    value={analysis.endDate}
                    onChange={(e: ChangeEvent<HTMLInputElement>) =>
                      setAnalysis({ ...analysis, endDate: e.target.value })
                    }
                    InputLabelProps={{ shrink: true }}
                    sx={{ flex: 1 }}
                    aria-label="End date for carry analysis"
                  />
                  <Button
                    variant="contained"
                    onClick={handleAnalyzeCarry}
                    disabled={oracleIsStale && !isManualSource}
                    // sx={{ alignSelf: { xs: "stretch", sm: "flex-start" } }}
                    aria-label="Analyze carry"
                    sx={{
                bgcolor: "#0b3d91",
                textTransform: "none",
                fontSize: 14,fontWeight:"bold",
                color: "#FFFFFF",
                "&:hover": { bgcolor: "#083475" },
              }}
                  >
                    Analyze Carry
                  </Button>
                </Box>
                {analysis.results === "loading" ? (
                  <Typography variant="body1">Loading analysis...</Typography>
                ) : (
                  analysis.results && (
                    <Box
                      sx={{
                        display: "grid",
                        gap: 2,
                        gridTemplateColumns: {
                          xs: "1fr",
                          sm: "1fr 1fr",
                          md: "repeat(4, 1fr)",
                        },
                      }}
                    >
                      <Paper
                        elevation={0}
                        sx={{
                          p: 2,
                          textAlign: "center",
                          bgcolor: "#e3f2fd",
                          color: "#0d47a1",
                        }}
                      >
                        <Typography
                          variant="body2"
                          sx={{ fontWeight: "medium" }}
                        >
                          Price Spread
                        </Typography>
                        <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                          {formatCurrency(analysis.results.priceSpread, 2)}
                        </Typography>
                      </Paper>
                      <Paper
                        elevation={0}
                        sx={{
                          p: 2,
                          textAlign: "center",
                          bgcolor: "#e8f5e9",
                          color: "#1b5e20",
                        }}
                      >
                        <Typography
                          variant="body2"
                          sx={{ fontWeight: "medium" }}
                        >
                          Basis Points
                        </Typography>
                        <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                          {`${analysis.results.basisPoints.toFixed(2)} bps`}
                        </Typography>
                      </Paper>
                      <Paper
                        elevation={0}
                        sx={{
                          p: 2,
                          textAlign: "center",
                          bgcolor: "#ede7f6",
                          color: "#311b92",
                        }}
                      >
                        <Typography
                          variant="body2"
                          sx={{ fontWeight: "medium" }}
                        >
                          Annualized Yield
                        </Typography>
                        <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                          {`${analysis.results.annualizedYield.toFixed(2)}%`}
                        </Typography>
                      </Paper>
                      <Paper
                        elevation={0}
                        sx={{
                          p: 2,
                          textAlign: "center",
                          bgcolor: theme.palette.grey[200],
                          color: "text.primary",
                        }}
                      >
                        <Typography
                          variant="body2"
                          sx={{ fontWeight: "medium" }}
                        >
                          Start / End Price
                        </Typography>
                        <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                          {analysis.results.startEndPrice}
                        </Typography>
                      </Paper>
                    </Box>
                  )
                )}
              </Paper>
              <Divider sx={{ my: 4 }} />
              <Box
                sx={{
                  display: "flex",
                  flexDirection: { xs: "column", md: "row" },
                  gap: 4,
                }}
              >
                <Paper
                  sx={{
                    p: 4,
                    flex: 1,
                    border: "1px solid #D4DBE3",
                    boxShadow: "none",
                  }}
                >
                  <Typography
                    variant="h6"
                    sx={{ mb: 2, color: theme.palette.text.primary }}
                  >
                    Outright Trade (Long Only)
                  </Typography>
                  <TextField
                    label="Maturity Date"
                    type="date"
                    value={trade.outright.maturity}
                    onChange={(e: ChangeEvent<HTMLInputElement>) =>
                      setTrade({
                        ...trade,
                        outright: {
                          ...trade.outright,
                          maturity: e.target.value,
                        },
                      })
                    }
                    InputLabelProps={{ shrink: true }}
                    fullWidth
                    sx={{ mb: 2 }}
                    aria-label="Maturity date for outright trade"
                  />
                  <TextField
                    label="Amount (IGLD)"
                    type="number"
                    value={trade.outright.amount}
                    onChange={(e: ChangeEvent<HTMLInputElement>) =>
                      setTrade({
                        ...trade,
                        outright: { ...trade.outright, amount: e.target.value },
                      })
                    }
                    placeholder="e.g., 1.0"
                    fullWidth
                    sx={{ mb: 2 }}
                    inputProps={{ min: 0, step: 0.1 }}
                    aria-label="Amount for outright trade"
                  />
                  <Box sx={{ mb: 2 }}>
                    <Typography variant="body2" sx={{ mb: 1 }}>
                      Max Cost (with slippage)
                    </Typography>
                    <Box sx={{ display: "flex", gap: 2 }}>
                      <TextField
                        value={trade.outright.maxCost}
                        InputProps={{ readOnly: true }}
                        placeholder="e.g., 1000"
                        fullWidth
                        aria-label="Max cost for outright trade (calculated)"
                      />
                      <Tooltip title="Percentage allowance for price changes during transaction">
                        <TextField
                          type="number"
                          value={trade.outright.slippagePercent}
                          onChange={(e: ChangeEvent<HTMLInputElement>) =>
                            setTrade({
                              ...trade,
                              outright: {
                                ...trade.outright,
                                slippagePercent: e.target.value,
                              },
                            })
                          }
                          InputProps={{
                            endAdornment: "%",
                            inputProps: { min: 0, max: 10, step: 0.1 },
                          }}
                          sx={{ width: 120 }}
                          aria-label="Slippage percentage for outright trade"
                        />
                      </Tooltip>
                    </Box>
                    <FormHelperText>
                      Max cost is automatically calculated based on the amount
                      and futures price, including slippage.
                    </FormHelperText>
                  </Box>
                  <FormControl fullWidth sx={{ mb: 2 }}>
                    <InputLabel id="outright-payment-token-label">
                      Pay With
                    </InputLabel>
                    <Select
                      labelId="outright-payment-token-label"
                      value={trade.outright.paymentToken}
                      onChange={(e: SelectChangeEvent) =>
                        setTrade({
                          ...trade,
                          outright: {
                            ...trade.outright,
                            paymentToken: e.target.value,
                          },
                        })
                      }
                      label="Pay With"
                      aria-label="Payment token for outright trade"
                    >
                      <MenuItem value={IUSD_ADDR}>IUSD</MenuItem>
                      <MenuItem value={ADDRESS_ZERO}>ETH</MenuItem>
                    </Select>
                  </FormControl>
                  <Button
                    variant="contained"
                    onClick={handleExecuteOutrightTrade}
                    disabled={
                      (oracleIsStale && !isManualSource) || isOutrightTrading
                    }
                    sx={{
                      py: 1.5,
                      bgcolor: "#0b3d91",
                      textTransform: "none",
                      fontWeight: "bold",
                      fontSize: 14,
                      "&:hover": { bgcolor: "#083475" },
                    }}
                    aria-label="Execute outright trade"
                  >
                    Execute Outright Trade
                  </Button>
                </Paper>
                <Paper
                  sx={{
                    p: 4,
                    flex: 1,
                    border: "1px solid #D4DBE3",
                    boxShadow: "none",
                  }}
                >
                  <Typography
                    variant="h6"
                    sx={{ mb: 2, color: theme.palette.text.primary }}
                  >
                    Multi-Leg Spread Trade
                  </Typography>
                  <Box sx={{ mb: 2 }}>
                    <Tooltip title="Simulate implied volatility for frontend analysis (not used on-chain)">
                      <Box
                        sx={{ display: "flex", alignItems: "center", gap: 2 }}
                      >
                        <FormControlLabel
                          control={
                            <Checkbox
                              checked={trade.spread.ivEnabled}
                              onChange={(e: ChangeEvent<HTMLInputElement>) =>
                                setTrade({
                                  ...trade,
                                  spread: {
                                    ...trade.spread,
                                    ivEnabled: e.target.checked,
                                  },
                                })
                              }
                              aria-label="Enable implied volatility simulation"
                            />
                          }
                          label="Implied Volatility (IV) Simulation"
                        />
                        <TextField
                          type="number"
                          value={trade.spread.impliedVolatility}
                          onChange={(e: ChangeEvent<HTMLInputElement>) =>
                            setTrade({
                              ...trade,
                              spread: {
                                ...trade.spread,
                                impliedVolatility: e.target.value,
                              },
                            })
                          }
                          disabled={!trade.spread.ivEnabled}
                          InputProps={{
                            endAdornment: "%",
                            inputProps: { min: 0, step: 0.1 },
                          }}
                          sx={{ width: 120 }}
                          aria-label="Implied volatility for spread trade"
                        />
                      </Box>
                    </Tooltip>
                    <Box sx={{ flex: 1, minWidth: 240, mt: 1 }}>
                      <Slider
                        aria-label="Implied volatility slider"
                        disabled={!trade.spread.ivEnabled}
                        value={Number(trade.spread.impliedVolatility || 0)}
                        onChange={(_, val) => {
                          const iv = Array.isArray(val) ? val[0] : val;
                          const nextIv =
                            typeof trade.spread.impliedVolatility === "number"
                              ? iv
                              : String(iv);
                          setTrade({
                            ...trade,
                            spread: {
                              ...trade.spread,
                              impliedVolatility: nextIv,
                            },
                          });
                        }}
                        step={0.1}
                        min={0}
                        max={200}
                        valueLabelDisplay="auto"
                      />
                    </Box>

                    <FormHelperText>
                      Note: IV is for frontend analysis only. The final on-chain
                      transaction price is determined by the contract's fixed
                      cost-of-carry model.
                    </FormHelperText>
                  </Box>
                  {trade.spread.legs.map((leg, index) => (
                    <Box
                      key={index}
                      sx={{
                        display: "flex",
                        flexDirection: { xs: "column", sm: "row" },
                        gap: 2,
                        alignItems: { sm: "center" },
                        mb: 2,
                      }}
                    >
                      <FormControl sx={{ width: { xs: "100%", sm: 120 } }}>
                        <InputLabel id={`leg-direction-${index}`}>
                          Direction
                        </InputLabel>
                        <Select
                          labelId={`leg-direction-${index}`}
                          value={leg.isLong ? "true" : "false"}
                          onChange={(e: SelectChangeEvent) => {
                            const newLegs = [...trade.spread.legs];
                            newLegs[index].isLong = e.target.value === "true";
                            setTrade({
                              ...trade,
                              spread: { ...trade.spread, legs: newLegs },
                            });
                          }}
                          label="Direction"
                          aria-label={`Direction for leg ${index + 1}`}
                        >
                          <MenuItem value="true">Long</MenuItem>
                          <MenuItem value="false">Short</MenuItem>
                        </Select>
                      </FormControl>
                      <Tooltip title="Select the maturity date for this leg">
                        <TextField
                          label="Maturity"
                          type="date"
                          value={leg.maturity}
                          onChange={(e: ChangeEvent<HTMLInputElement>) => {
                            const newLegs = [...trade.spread.legs];
                            newLegs[index].maturity = e.target.value;
                            setTrade({
                              ...trade,
                              spread: { ...trade.spread, legs: newLegs },
                            });
                          }}
                          InputLabelProps={{ shrink: true }}
                          sx={{ flex: 1 }}
                          aria-label={`Maturity date for leg ${index + 1}`}
                        />
                      </Tooltip>
                      <Tooltip title="Enter the amount of gold tokens (IGLD) for this leg">
                        <TextField
                          label="Amount (IGLD)"
                          type="number"
                          value={leg.amountIGLD}
                          onChange={(e: ChangeEvent<HTMLInputElement>) => {
                            const newLegs = [...trade.spread.legs];
                            newLegs[index].amountIGLD = e.target.value;
                            setTrade({
                              ...trade,
                              spread: { ...trade.spread, legs: newLegs },
                            });
                          }}
                          placeholder="e.g., 1.0"
                          sx={{ flex: 1 }}
                          inputProps={{ min: 0, step: 0.1 }}
                          aria-label={`Amount for leg ${index + 1}`}
                        />
                      </Tooltip>
                      {index > 1 && (
                        <Button
                          variant="outlined"
                          color="error"
                          onClick={() => {
                            const newLegs = trade.spread.legs.filter(
                              (_, i) => i !== index
                            );
                            setTrade({
                              ...trade,
                              spread: { ...trade.spread, legs: newLegs },
                            });
                          }}
                          aria-label={`Remove leg ${index + 1}`}
                        >
                          ×
                        </Button>
                      )}
                    </Box>
                  ))}
                  <Button
                    variant="outlined"
                    onClick={() =>
                      setTrade({
                        ...trade,
                        spread: {
                          ...trade.spread,
                          legs: [
                            ...trade.spread.legs,
                            { maturity: "", amountIGLD: "", isLong: true },
                          ],
                        },
                      })
                    }
                    sx={{ alignSelf: "flex-start", mb: 2 }}
                    aria-label="Add new leg to spread trade"
                  >
                    Add Leg
                  </Button>
                  <FormControl fullWidth sx={{ mb: 2 }}>
                    <InputLabel id="spread-payment-token-label">
                      Pay With / Receive In
                    </InputLabel>
                    <Select
                      labelId="spread-payment-token-label"
                      value={trade.spread.paymentToken}
                      onChange={(e: SelectChangeEvent) =>
                        setTrade({
                          ...trade,
                          spread: {
                            ...trade.spread,
                            paymentToken: e.target.value,
                          },
                        })
                      }
                      label="Pay With / Receive In"
                      aria-label="Payment token for spread trade"
                    >
                      <MenuItem value={IUSD_ADDR}>IUSD</MenuItem>
                      <MenuItem value={ADDRESS_ZERO}>ETH</MenuItem>
                    </Select>
                  </FormControl>
                  <Box sx={{ mb: 2 }}>
                    <Typography variant="body2" sx={{ mb: 1 }}>
                      Max Cost / Min Credit (with slippage)
                    </Typography>
                    <Box
                      sx={{
                        display: "flex",
                        flexDirection: { xs: "column", sm: "row" },
                        gap: 2,
                        alignItems: { sm: "center" },
                      }}
                    >
                      <Tooltip title="Maximum cost or minimum credit for the spread trade">
                        <TextField
                          value={trade.spread.maxCost || trade.spread.minCredit}
                          InputProps={{ readOnly: true }}
                          placeholder="e.g., 1000"
                          fullWidth
                          aria-label="Max cost or min credit for spread trade"
                        />
                      </Tooltip>
                      <Tooltip title="Percentage allowance for price changes during transaction">
                        <TextField
                          type="number"
                          value={trade.spread.slippagePercent}
                          onChange={(e: ChangeEvent<HTMLInputElement>) =>
                            setTrade({
                              ...trade,
                              spread: {
                                ...trade.spread,
                                slippagePercent: e.target.value,
                              },
                            })
                          }
                          InputProps={{
                            endAdornment: "%",
                            inputProps: { min: 0, max: 10, step: 0.1 },
                          }}
                          sx={{ width: { xs: "100%", sm: 120 } }}
                          aria-label="Slippage percentage for spread trade"
                        />
                      </Tooltip>
                    </Box>
                    <FormHelperText>
                      Transaction will fail if the cost exceeds this amount due
                      to price changes.
                    </FormHelperText>
                    <Box sx={{ mb: 0 }}>
                      {!spreadQuote.isCalculating && spreadQuote.error && (
                        <Typography
                          variant="body2"
                          sx={{
                            color: theme.palette.error.main,
                            textAlign: "center",
                          }}
                        >
                          {spreadQuote.error}
                        </Typography>
                      )}
                      {!spreadQuote.error && spreadQuote.netCost !== null && (
                        <Paper
                          elevation={0}
                          sx={{
                            p: 2,
                            border: "1px solid #D4DBE3",
                            borderRadius: 1,
                            textAlign: "center",
                            backgroundColor: "#F7FAFC",
                          }}
                        >
                          <Typography
                            variant="body2"
                            sx={{
                              color: theme.palette.text.secondary,
                              textAlign: "center",
                              mb: 1,
                            }}
                          >
                            Estimated Net Cost/Credit (with IV)
                          </Typography>

                          {spreadQuote.isCalculating ? (
                            <Typography variant="body2">
                              Calculating...
                            </Typography>
                          ) : (
                            <Typography
                              variant="body1"
                              sx={{
                                fontWeight: "bold",
                                color: theme.palette.text.primary,
                                textAlign: "center",
                              }}
                            >
                              {spreadQuote.netCost > BigInt(0)
                                ? "Debit:"
                                : "Debit:"}{" "}
                              $
                              {parseFloat(
                                spreadQuote.displayCost.replace(/[^0-9.]/g, "")
                              ).toFixed(2)}
                            </Typography>
                          )}
                        </Paper>
                      )}
                    </Box>
                  </Box>
                  <Button
                    variant="contained"
                    onClick={handleExecuteSpreadTrade}
                    disabled={
                      (oracleIsStale && !isManualSource) || isSpreadTrading
                    }
                    sx={{
                      py: 1.5,
                      bgcolor: "#0b3d91",
                      textTransform: "none",
                      fontWeight: "bold",
                      fontSize: "14px",
                      "&:hover": { bgcolor: "#083475" },
                    }}
                    aria-label="Execute spread trade"
                  >
                    Execute Spread Trade
                  </Button>
                </Paper>
              </Box>
            </Box>
          )}

          {activeTab === "admin" && isAdmin && (
            <Box>
              <Typography
                variant="h5"
                sx={{
                  fontWeight: "bold",
                  mb: 3,
                  color: theme.palette.text.primary,
                }}
              >
                Admin Controls
              </Typography>
              <Paper
                sx={{
                  p: 4,
                  mb: 4,
                  border: "1px solid #D4DBE3",
                  boxShadow: "none",
                }}
              >
                <Typography
                  variant="h6"
                  sx={{ mb: 2, color: theme.palette.text.primary }}
                >
                  Pricing Controls
                </Typography>
                <Box sx={{ display: "flex", flexDirection: "column", gap: 3 }}>
                  <FormControl fullWidth>
                    <InputLabel id="spot-price-source-label">
                      Spot Price Source
                    </InputLabel>
                    <Select
                      labelId="spot-price-source-label"
                      value={isManualSource ? "Manual" : "Chainlink"}
                      onChange={(e: SelectChangeEvent) => {
                        void handleSetPriceSource(
                          e.target.value as "Manual" | "Chainlink"
                        );
                      }}
                      label="Spot Price Source"
                      aria-label="Select spot price source"
                    >
                      <MenuItem value="Chainlink">Chainlink</MenuItem>
                      <MenuItem value="Manual">Manual</MenuItem>
                    </Select>
                  </FormControl>
                  {isManualSource && (
                    <>
                      <Tooltip title="Set the manual spot price for gold in USD">
                        <TextField
                          label="Manual Spot Price (USD)"
                          type="number"
                          value={admin.manualSpotPrice}
                          onChange={(e: ChangeEvent<HTMLInputElement>) =>
                            setAdmin({
                              ...admin,
                              manualSpotPrice: e.target.value,
                            })
                          }
                          placeholder="e.g., 2000"
                          fullWidth
                          inputProps={{ min: 0, step: 0.01 }}
                          aria-label="Manual spot price for gold"
                        />
                      </Tooltip>
                      <Button
                        variant="contained"
                        onClick={handleSetManualSpotPrice}
                        disabled={
                          !admin.manualSpotPrice ||
                          Number(admin.manualSpotPrice) <= 0
                        }
                        sx={{ alignSelf: "flex-start", mt: 1 }}
                        aria-label="Set manual spot price"
                      >
                        Set Price
                      </Button>
                    </>
                  )}
                </Box>
              </Paper>
              <Paper
                sx={{ p: 4, border: "1px solid #D4DBE3", boxShadow: "none" }}
              >
                <Typography
                  variant="h6"
                  sx={{ mb: 2, color: theme.palette.text.primary }}
                >
                  System Settings
                </Typography>
                <Box sx={{ display: "flex", flexDirection: "column", gap: 3 }}>
                  <Tooltip title="Maximum age of oracle data in seconds before it's considered stale">
                    <TextField
                      label="Oracle Max Age (seconds)"
                      type="number"
                      value={admin.oracleMaxAge}
                      onChange={(e: ChangeEvent<HTMLInputElement>) =>
                        setAdmin({ ...admin, oracleMaxAge: e.target.value })
                      }
                      placeholder="e.g., 3600"
                      fullWidth
                      inputProps={{ min: 0 }}
                      aria-label="Oracle max age in seconds"
                    />
                  </Tooltip>
                  <Button
                    variant="contained"
                    onClick={handleSetOracleMaxAge}
                    sx={{ alignSelf: "flex-start" }}
                    aria-label="Set oracle max age"
                  >
                    Set Max Age
                  </Button>
                  <Tooltip title="Annual interest rate in basis points (1 BPS = 0.01%)">
                    <TextField
                      label="Annual Rate (BPS)"
                      type="number"
                      value={admin.annualRate}
                      onChange={(e: ChangeEvent<HTMLInputElement>) =>
                        setAdmin({ ...admin, annualRate: e.target.value })
                      }
                      placeholder="e.g., 500"
                      fullWidth
                      inputProps={{ min: 0 }}
                      aria-label="Annual rate in basis points"
                    />
                  </Tooltip>
                  <Button
                    variant="contained"
                    onClick={handleSetAnnualRateBPS}
                    sx={{ alignSelf: "flex-start" }}
                    aria-label="Set annual rate"
                  >
                    Set Rate
                  </Button>
                  <Tooltip title="Ethereum address of the keeper to manage the contract">
                    <TextField
                      label="Keeper Address"
                      value={admin.keeperAddress}
                      onChange={(e: ChangeEvent<HTMLInputElement>) =>
                        setAdmin({ ...admin, keeperAddress: e.target.value })
                      }
                      placeholder="e.g., 0x1234...abcd"
                      fullWidth
                      aria-label="Keeper address"
                    />
                  </Tooltip>
                  <Box
                    sx={{
                      display: "flex",
                      flexDirection: { xs: "column", sm: "row" },
                      gap: 2,
                    }}
                  >
                    <Button
                      variant="contained"
                      onClick={handleAddKeeper}
                      sx={{ flex: 1 }}
                      aria-label="Add keeper"
                    >
                      Add Keeper
                    </Button>
                    <Button
                      variant="contained"
                      color="secondary"
                      onClick={handleRemoveKeeper}
                      sx={{ flex: 1 }}
                      aria-label="Remove keeper"
                    >
                      Remove Keeper
                    </Button>
                  </Box>
                </Box>
              </Paper>
            </Box>
          )}
        </main>
      )}

      <ConfirmCloseDialog
        open={Boolean(pendingClosePosition)}
        title="Close position"
        description={
          pendingClosePosition
            ? `Are you sure you want to close position #${
                pendingClosePosition.positionId
              } (${
                pendingClosePosition.positionType
              })?\nCurrent P/L: ${formatPnlWithSign(pendingClosePosition.pnl)}`
            : ""
        }
        confirmLabel="Close position"
        cancelLabel="Keep open"
        onConfirm={handleConfirmClosePosition}
        onCancel={handleCancelClosePosition}
        loading={isCloseProcessing}
      />

      <Snackbar
        open={toast.visible}
        autoHideDuration={4000}
        onClose={() => setToast({ ...toast, visible: false })}
        anchorOrigin={{ vertical: "bottom", horizontal: "right" }}
        aria-label="Notification"
      >
        <Alert
          severity={toast.type}
          sx={{ width: "100%", border: "1px solid #D4DBE3", boxShadow: "none" }}
          aria-label={`Notification: ${toast.message}`}
        >
          {toast.message}
        </Alert>
      </Snackbar>
    </Box>
  );
};

export default IntelliTradeGoldWithoutSLTP;
